<?php
require_once '../config/db.php';
require_once '../config/functions.php';

requerirRol(['admin']);

$id = intval($_GET['id'] ?? 0);

if (!$id) {
    header('Location: index.php');
    exit;
}

// Obtener apoderado
$apoderado = query("SELECT a.*, u.usuario FROM apoderados a LEFT JOIN usuarios u ON a.id_usuario = u.id WHERE a.id = :id", ['id' => $id])->fetch();

if (!$apoderado) {
    header('Location: index.php');
    exit;
}

// Obtener año lectivo seleccionado
$anio_lectivo_id = getAnioLectivoSeleccionado();

// Verificar si tiene estudiantes asociados en el año lectivo actual
$estudiantes_asociados = query("
    SELECT e.id, e.dni, CONCAT(e.apellido_paterno, ' ', e.apellido_materno, ', ', e.nombres) as nombre,
           ea.parentesco, ea.es_principal
    FROM estudiantes_apoderados ea
    INNER JOIN estudiantes e ON ea.id_estudiante = e.id
    WHERE ea.id_apoderado = ? AND ea.estado = 'activo'
    AND ea.id_anio_lectivo = ?
    ORDER BY e.apellido_paterno, e.apellido_materno, e.nombres
", [$id, $anio_lectivo_id])->fetchAll();

$tiene_asignaciones = (count($estudiantes_asociados) > 0);

$mensaje = '';
$tipo_mensaje = '';
$eliminado = false;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $accion = $_POST['accion'] ?? '';

    if ($accion === 'eliminar') {
        try {
            // VALIDACIÓN: Verificar si tiene estudiantes asociados en el año lectivo actual
            if ($tiene_asignaciones) {
                $mensaje = "No se puede eliminar al apoderado porque tiene " . count($estudiantes_asociados) . " estudiante(s) asociado(s) en el año lectivo actual. Primero debe desvincular los estudiantes.";
                $tipo_mensaje = 'danger';
            } else {
                // Iniciar transacción
                $pdo->beginTransaction();

                // Primero eliminar el apoderado
                $stmt = $pdo->prepare("DELETE FROM apoderados WHERE id = ?");
                $stmt->execute([$id]);

                // Luego eliminar el usuario
                $stmt = $pdo->prepare("DELETE FROM usuarios WHERE id = ?");
                $stmt->execute([$apoderado['id_usuario']]);

                $pdo->commit();
                $mensaje = 'Apoderado eliminado exitosamente';
                $tipo_mensaje = 'success';
                $eliminado = true;
            }
        } catch (Exception $e) {
            if ($pdo->inTransaction()) {
                $pdo->rollBack();
            }
            $mensaje = 'Error al eliminar: ' . $e->getMessage();
            $tipo_mensaje = 'danger';
        }
    } elseif ($accion === 'cambiar_estado') {
        $nuevo_estado = $_POST['nuevo_estado'] ?? 'inactivo';
        try {
            // Iniciar transacción
            $pdo->beginTransaction();

            // Actualizar estado del apoderado
            $stmt = $pdo->prepare("UPDATE apoderados SET estado = ? WHERE id = ?");
            $stmt->execute([$nuevo_estado, $id]);

            // Actualizar estado del usuario
            $usuario_estado = ($nuevo_estado === 'activo') ? 'activo' : 'inactivo';
            $stmt = $pdo->prepare("UPDATE usuarios SET estado = ? WHERE id = ?");
            $stmt->execute([$usuario_estado, $apoderado['id_usuario']]);

            $pdo->commit();
            $mensaje = 'Estado actualizado exitosamente';
            $tipo_mensaje = 'success';

            // Recargar datos
            $stmt = $pdo->prepare("SELECT a.*, u.usuario FROM apoderados a LEFT JOIN usuarios u ON a.id_usuario = u.id WHERE a.id = ?");
            $stmt->execute([$id]);
            $apoderado = $stmt->fetch();
        } catch (Exception $e) {
            if ($pdo->inTransaction()) {
                $pdo->rollBack();
            }
            $mensaje = 'Error al cambiar estado: ' . $e->getMessage();
            $tipo_mensaje = 'danger';
        }
    }
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>SICA - Eliminar Apoderado</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css" rel="stylesheet">
    <link href="../assets/css/sica.css?v=5.1" rel="stylesheet">
</head>
<body>
    <nav class="sidebar">
        <?php include '../includes/sidebar.php'; ?>
    </nav>

    <div class="main-content">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <div>
                <h3 class="mb-1">Eliminar Apoderado</h3>
                <p class="text-muted mb-0">Gestión de eliminación de apoderados</p>
            </div>
            <a href="index.php" class="btn btn-outline-secondary">
                <i class="bi bi-arrow-left"></i> Volver
            </a>
        </div>

        <?php if ($mensaje): ?>
            <div class="alert alert-<?= $tipo_mensaje ?> alert-dismissible fade show" role="alert">
                <?= htmlspecialchars($mensaje) ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <?php if (!$eliminado): ?>
            <div class="row">
                <div class="col-lg-6">
                    <div class="card">
                        <div class="card-header bg-danger text-white">
                            <h5 class="mb-0"><i class="bi bi-exclamation-triangle"></i> Confirmar Eliminación</h5>
                        </div>
                        <div class="card-body">
                            <?php if ($tiene_asignaciones): ?>
                                <div class="alert alert-danger">
                                    <i class="bi bi-x-circle"></i>
                                    <strong>No se puede eliminar este apoderado</strong>
                                    <hr class="my-2">
                                    <p class="mb-1">El apoderado tiene las siguientes asignaciones en el año lectivo actual:</p>
                                    <ul class="mb-0">
                                        <li><strong><?= count($estudiantes_asociados) ?> estudiante(s) asociado(s)</strong></li>
                                    </ul>
                                    <?php if (count($estudiantes_asociados) <= 5): ?>
                                        <details class="mt-2">
                                            <summary class="small text-muted" style="cursor: pointer;">Ver lista de estudiantes</summary>
                                            <ul class="mb-0 mt-2 small">
                                                <?php foreach ($estudiantes_asociados as $est): ?>
                                                    <li>
                                                        <?= htmlspecialchars($est['nombre']) ?> (DNI: <?= htmlspecialchars($est['dni']) ?>)
                                                        - <?= htmlspecialchars($est['parentesco']) ?>
                                                        <?php if ($est['es_principal']): ?>
                                                            <span class="badge bg-warning">Principal</span>
                                                        <?php endif; ?>
                                                    </li>
                                                <?php endforeach; ?>
                                            </ul>
                                        </details>
                                    <?php endif; ?>
                                    <p class="mb-0 mt-2"><small>Debe desvincular estos estudiantes antes de eliminar.</small></p>
                                </div>
                            <?php else: ?>
                                <div class="alert alert-warning">
                                    <i class="bi bi-exclamation-circle"></i>
                                    <strong>Advertencia:</strong> Esta acción eliminará permanentemente al apoderado y su usuario del sistema. Esta acción no se puede deshacer.
                                </div>
                            <?php endif; ?>

                            <div class="mb-4">
                                <h6>Apoderado a eliminar:</h6>
                                <table class="table table-bordered">
                                    <tr>
                                        <td width="30%"><strong>DNI:</strong></td>
                                        <td><?= htmlspecialchars($apoderado['dni']) ?></td>
                                    </tr>
                                    <tr>
                                        <td><strong>Nombre:</strong></td>
                                        <td>
                                            <?= htmlspecialchars($apoderado['apellido_paterno'] . ' ' . $apoderado['apellido_materno'] . ', ' . $apoderado['nombres']) ?>
                                        </td>
                                    </tr>
                                    <tr>
                                        <td><strong>Usuario:</strong></td>
                                        <td><?= htmlspecialchars($apoderado['usuario']) ?></td>
                                    </tr>
                                    <tr>
                                        <td><strong>Estado:</strong></td>
                                        <td>
                                            <span class="badge <?= $apoderado['estado'] === 'activo' ? 'bg-success' : 'bg-secondary' ?>">
                                                <?= ucfirst($apoderado['estado']) ?>
                                            </span>
                                        </td>
                                    </tr>
                                </table>
                            </div>

                            <form method="POST">
                                <input type="hidden" name="accion" value="eliminar">
                                <div class="d-flex gap-2">
                                    <button type="submit" class="btn btn-danger" <?= $tiene_asignaciones ? 'disabled' : '' ?>>
                                        <i class="bi bi-trash"></i> Sí, Eliminar Apoderado
                                    </button>
                                    <a href="index.php" class="btn btn-secondary">
                                        <i class="bi bi-x-circle"></i> Cancelar
                                    </a>
                                </div>
                                <?php if ($tiene_asignaciones): ?>
                                    <div class="form-text text-muted mt-2">
                                        <i class="bi bi-info-circle"></i> El botón está deshabilitado porque el apoderado tiene estudiantes asociados.
                                    </div>
                                <?php endif; ?>
                            </form>
                        </div>
                    </div>
                </div>

                <div class="col-lg-6">
                    <div class="card mb-3">
                        <div class="card-header bg-warning">
                            <h5 class="mb-0"><i class="bi bi-lightbulb"></i> Alternativas</h5>
                        </div>
                        <div class="card-body">
                            <p>En lugar de eliminar al apoderado, considere estas opciones:</p>

                            <h6>Cambiar a Inactivo</h6>
                            <p class="small text-muted">Mantiene los datos pero impide el acceso al sistema.</p>
                            <form method="POST" class="mb-3">
                                <input type="hidden" name="accion" value="cambiar_estado">
                                <input type="hidden" name="nuevo_estado" value="inactivo">
                                <button type="submit" class="btn btn-outline-warning">
                                    <i class="bi bi-dash-circle"></i> Cambiar a Inactivo
                                </button>
                            </form>

                            <h6>Cambiar a Retirado</h6>
                            <p class="small text-muted">Marca al apoderado como retirado de la institución.</p>
                            <form method="POST">
                                <input type="hidden" name="accion" value="cambiar_estado">
                                <input type="hidden" name="nuevo_estado" value="retirado">
                                <button type="submit" class="btn btn-outline-secondary">
                                    <i class="bi bi-person-x"></i> Cambiar a Retirado
                                </button>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
        <?php else: ?>
            <div class="text-center py-5">
                <i class="bi bi-check-circle text-success" style="font-size: 4rem;"></i>
                <h4 class="mt-3">Apoderado Eliminado</h4>
                <p class="text-muted">El apoderado ha sido eliminado exitosamente del sistema.</p>
                <a href="index.php" class="btn btn-primary">
                    <i class="bi bi-arrow-left"></i> Volver al Listado
                </a>
            </div>
        <?php endif; ?>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
