<?php
require_once '../config/db.php';
require_once '../config/functions.php';

requerirRol(['admin', 'auxiliar', 'docente']);

$anio_activo = query("SELECT id, anio FROM anios_lectivos WHERE estado = 'activo' LIMIT 1")->fetch();
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>SICA - Escanear QR</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css" rel="stylesheet">
    <link href="../assets/css/sica.css?v=5.1" rel="stylesheet">
    <script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
    <script src="https://unpkg.com/html5-qrcode@2.3.8/html5-qrcode.min.js"></script>
    <style>
        .qr-container {
            max-width: 600px;
            margin: 0 auto;
        }
        .scanner-card {
            background: white;
            border-radius: 20px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.1);
            padding: 30px;
        }
        #reader {
            border-radius: 15px;
            overflow: hidden;
        }
        #reader video {
            border-radius: 15px;
        }
        #reader__scan_region {
            border-radius: 15px;
        }
        .result-card {
            background: white;
            border-radius: 15px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.05);
            padding: 25px;
            margin-top: 20px;
            display: none;
        }
        .result-card.show {
            display: block;
            animation: slideIn 0.3s ease-out;
        }
        @keyframes slideIn {
            from {
                opacity: 0;
                transform: translateY(-10px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        .scanner-icon {
            font-size: 4rem;
            color: var(--primary-color);
            margin-bottom: 20px;
        }
        .btn-scan {
            background: linear-gradient(135deg, var(--primary-color), #3b82f6);
            border: none;
            border-radius: 12px;
            padding: 15px 40px;
            font-size: 1.1rem;
            font-weight: 600;
            transition: all 0.3s;
        }
        .btn-scan:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 25px rgba(30, 64, 175, 0.3);
        }
        .recent-list {
            max-height: 400px;
            overflow-y: auto;
        }
        .recent-item {
            transition: all 0.2s;
        }
        .recent-item:hover {
            background-color: #f8f9fa;
        }
        .estado-badge {
            font-size: 0.9rem;
            padding: 0.5rem 0.75rem;
            border-radius: 8px;
        }
        .result-card.success {
            background: linear-gradient(135deg, #d1fae5 0%, #a7f3d0 100%);
            border-left: 5px solid #10b981;
        }
        .result-card.warning {
            background: linear-gradient(135deg, #fed7aa 0%, #fdba74 100%);
            border-left: 5px solid #f97316;
        }
        .result-card.error {
            background: linear-gradient(135deg, #fecaca 0%, #fca5a5 100%);
            border-left: 5px solid #ef4444;
        }
    </style>
</head>
<body>
    <?php include '../includes/layout_elems.php'; ?>

    <div class="d-flex justify-content-between align-items-center mb-4">
        <div class="d-flex align-items-center">
            <button class="btn btn-link d-lg-none me-3" onclick="toggleSidebarMobile()">
                <i class="bi bi-list fs-4"></i>
            </button>
            <div>
                <h3 class="mb-1">Escaneo de Código QR</h3>
                <p class="text-muted mb-0">Registro de asistencias por escaneo - <?= htmlspecialchars($anio_activo['anio']) ?></p>
            </div>
        </div>
        <div>
            <a href="index.php" class="btn btn-outline-secondary me-2">
                <i class="bi bi-calendar-check-fill me-2"></i>Ver Asistencias
            </a>
            <a href="../dashboard.php" class="btn btn-outline-primary">
                <i class="bi bi-house-fill me-2"></i>Dashboard
            </a>
        </div>
    </div>

    <div class="row">
        <div class="col-lg-6">
            <div class="qr-container">
                <div class="scanner-card text-center">
                    <div class="scanner-icon">
                        <i class="bi bi-qr-code-scan"></i>
                    </div>
                    <h4 class="mb-3">Escáner de Asistencias</h4>
                    <p class="text-muted mb-4">Apunte la cámara hacia el código QR del estudiante</p>

                    <div id="reader" class="mb-4"></div>

                    <div class="d-flex gap-2 justify-content-center">
                        <button onclick="iniciarEscanner()" class="btn btn-scan text-white" id="btnIniciar">
                            <i class="bi bi-camera-fill me-2"></i>Iniciar Cámara
                        </button>
                        <button onclick="detenerEscanner()" class="btn btn-danger" id="btnDetener" style="display: none;">
                            <i class="bi bi-stop-circle-fill me-2"></i>Detener
                        </button>
                    </div>

                    <div class="mt-4">
                        <a href="registrar_manual.php" class="text-decoration-none d-block mb-3">
                            <i class="bi bi-keyboard-fill me-2"></i>Registro Manual por DNI
                        </a>
                        <a href="diagnosticar_camara.php" class="text-decoration-none d-block">
                            <i class="bi bi-tools me-2"></i>Diagnóstico de Cámara
                        </a>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-lg-6">
            <!-- Resultado del escaneo -->
            <div id="resultCard" class="result-card">
                <div class="d-flex align-items-center mb-3">
                    <div id="resultIcon" class="me-3 fs-2"></div>
                    <div>
                        <h5 class="mb-0" id="resultTitle">Resultado del Escaneo</h5>
                        <p class="text-muted mb-0 small" id="resultTime"></p>
                    </div>
                </div>
                <hr>
                <div id="resultContent"></div>
            </div>

            <!-- Registros recientes -->
            <div class="card mt-4">
                <div class="card-header bg-white">
                    <h6 class="mb-0"><i class="bi bi-clock-history me-2"></i>Registros Recientes</h6>
                </div>
                <div class="card-body recent-list" id="recentList">
                    <p class="text-muted text-center">No hay registros recientes</p>
                </div>
            </div>
        </div>
    </div>

    <?php include '../includes/layout_scripts.php'; ?>

    <script>
        let html5QrcodeScanner = null;
        let isScanning = false;
        const recientes = [];

        // Lista temporal de DNIs recientemente registrados (para evitar escaneos duplicados muy rápidos)
        const dnisRecientes = new Map(); // Mapa: DNI -> timestamp
        const TIEMPO_BLOQUEO_DNI = 3000; // 3 segundos de bloqueo después de un registro exitoso

        // Función para limpiar DNIs antiguos del mapa (se ejecuta periódicamente)
        function limpiarDnisAntiguos() {
            const ahora = Date.now();
            for (const [dni, timestamp] of dnisRecientes.entries()) {
                if (ahora - timestamp > TIEMPO_BLOQUEO_DNI) {
                    dnisRecientes.delete(dni);
                }
            }
        }

        // Limpiar cada 30 segundos automáticamente
        setInterval(limpiarDnisAntiguos, 30000);

        // Verificar si estamos en HTTPS
        function verificarHTTPS() {
            if (location.protocol !== 'https:' && location.hostname !== 'localhost' && location.hostname !== '127.0.0.1') {
                mostrarResultado('error', 'Error de Seguridad',
                    '<strong>Se requiere HTTPS</strong><br>Los navegadores modernos requieren conexión segura para acceder a la cámara.<br><small>Use https://localhost o configure HTTPS en el servidor.</small>');
                $('#btnIniciar').prop('disabled', true).html('<i class="bi bi-lock-fill me-2"></i>HTTPS Requerido');
                return false;
            }
            return true;
        }

        // Verificar permisos de cámara al cargar la página
        function verificarPermisosCamara() {
            if (!navigator.mediaDevices || !navigator.mediaDevices.getUserMedia) {
                mostrarResultado('error', 'Cámara No Disponible',
                    '<strong>Browser incompatible</strong><br>Su navegador no soporta acceso a la cámara.<br><small>Use Chrome, Firefox, Edge o Safari.</small>');
                $('#btnIniciar').prop('disabled', true);
                return false;
            }
            return true;
        }

        function iniciarEscanner() {
            // Verificaciones previas
            if (!verificarHTTPS()) return;
            if (!verificarPermisosCamara()) return;

            $('#btnIniciar').hide();
            $('#btnDetener').show();

            // Mostrar mensaje de "Iniciando..."
            mostrarResultado('warning', 'Iniciando Cámara', '<small>Por favor espere...</small>');

            html5QrcodeScanner = new Html5Qrcode("reader");
            const config = {
                fps: 10,
                qrbox: { width: 250, height: 250 },
                aspectRatio: 1.0
            };

            // Iniciar escáner directamente (la librería maneja los permisos)
            html5QrcodeScanner.start(
                { facingMode: "environment" },
                config,
                onScanSuccess,
                onScanFailure
            ).then(() => {
                isScanning = true;
                console.log("Escáner iniciado correctamente");
                // Ocultar la tarjeta de resultado cuando se inicia correctamente
                $('#resultCard').removeClass('show');
            }).catch(err => {
                console.error("Error al iniciar escáner:", err);

                // Manejo específico de errores
                if (err.name === 'NotAllowedError') {
                    mostrarResultado('error', 'Permiso Denegado',
                        '<strong>Acceso a la cámara denegado</strong><br><br>' +
                        '<small>Para solucionar:<br>' +
                        '1. Haga clic en el icono de candado 🔒 en la barra de direcciones<br>' +
                        '2. Permita acceso a la cámara<br>' +
                        '3. Recargue la página y vuelva a intentar</small>');
                } else if (err.name === 'NotFoundError') {
                    mostrarResultado('error', 'Cámara No Encontrada',
                        '<strong>No se detectó ninguna cámara</strong><br><small>Verifique que su cámara esté conectada</small>');
                } else if (err.name === 'NotReadableError') {
                    mostrarResultado('error', 'Cámara en Uso',
                        '<strong>La cámara está siendo usada por otra aplicación</strong><br><small>Cierre Zoom, Teams, Meet, etc.</small>');
                } else {
                    mostrarResultado('error', 'Error al Iniciar',
                        '<strong>No se pudo iniciar el escáner</strong><br><small>' + err + '</small>');
                }

                $('#btnIniciar').show();
                $('#btnDetener').hide();
            });
        }

        function detenerEscanner() {
            if (html5QrcodeScanner && isScanning) {
                html5QrcodeScanner.stop().then(() => {
                    isScanning = false;
                    $('#btnIniciar').show();
                    $('#btnDetener').hide();
                }).catch(err => {
                    console.error("Error al detener:", err);
                });
            }
        }

        function onScanSuccess(decodedText, decodedResult) {
            console.log("QR escaneado:", decodedText);

            // Decodificar el DNI (está en base64)
            let dni = decodedText;

            // Intentar decodificar si está en base64
            try {
                const decoded = atob(decodedText);
                if (/^\d{8}$/.test(decoded)) {
                    dni = decoded;
                }
            } catch (e) {
                // Si no es base64, usar el texto tal cual
            }

            // Validar que sea un DNI válido
            if (!/^\d{8}$/.test(dni)) {
                mostrarResultado('error', 'DNI Inválido', 'Código QR no válido');
                playErrorBeep();
                return;
            }

            // VERIFICAR BLOQUEO TEMPORAL (evita escaneos duplicados muy rápidos)
            const ahora = Date.now();
            if (dnisRecientes.has(dni)) {
                const tiempoRegistro = dnisRecientes.get(dni);
                const tiempoTranscurrido = ahora - tiempoRegistro;

                if (tiempoTranscurrido < TIEMPO_BLOQUEO_DNI) {
                    // Aún está dentro del tiempo de bloqueo
                    const tiempoRestante = Math.ceil((TIEMPO_BLOQUEO_DNI - tiempoTranscurrido) / 1000);
                    mostrarResultado(
                        'warning',
                        '⏱ Espera un momento',
                        `<small>Este estudiante ya fue registrado hace unos segundos.</small><br><small class="text-muted">Espere ${tiempoRestante} segundos antes de escanearlo nuevamente.</small>`
                    );
                    playWarningBeep();

                    // Pausar brevemente
                    if (html5QrcodeScanner) {
                        html5QrcodeScanner.pause();
                        setTimeout(() => {
                            if (isScanning) html5QrcodeScanner.resume();
                        }, 1500);
                    }
                    return;
                } else {
                    // Ya pasó el tiempo de bloqueo, eliminar del mapa
                    dnisRecientes.delete(dni);
                }
            }

            // Registrar asistencia
            $.post('/SICA/config/ajax.php', {
                action: 'registrar_asistencia',
                dni: dni,
                metodo: 'qr'
            }, function(response) {
                if (response.success) {
                    // ESTADO VERDE: Registro exitoso
                    mostrarResultado(
                        'success',
                        '✓ Registrado',
                        `${response.estudiante.nombre}<br><small class="text-muted">${response.estudiante.dni}</small><br><small class="text-muted">${response.estudiante.grado} - "${response.estudiante.seccion}"</small>`
                    );

                    agregarARecientes(response.estudiante);
                    playBeep();

                    // AGREGAR A LISTA DE BLOQUEO TEMPORAL
                    dnisRecientes.set(dni, Date.now());

                    // Pausar escaneo solo 1 segundo (para cola rápida)
                    if (html5QrcodeScanner) {
                        html5QrcodeScanner.pause();
                        setTimeout(() => {
                            if (isScanning) html5QrcodeScanner.resume();
                        }, 1000);
                    }
                } else {
                    // ESTADO NARANJA: Duplicado (ya registró hoy)
                    if (response.mensaje && response.mensaje.includes('ya tiene asistencia registrada hoy')) {
                        mostrarResultado(
                            'warning',
                            '⚠ Duplicado',
                            `${response.estudiante?.nombre || 'Estudiante'}<br><small class="text-muted">${response.estudiante?.dni || dni}</small><br><small class="text-muted">${response.estudiante?.grado || ''} - "${response.estudiante?.seccion || ''}"</small><br><small>Ya registró asistencia</small>`
                        );
                        playWarningBeep();

                        // Pausar 1.5 segundos
                        if (html5QrcodeScanner) {
                            html5QrcodeScanner.pause();
                            setTimeout(() => {
                                if (isScanning) html5QrcodeScanner.resume();
                            }, 1500);
                        }
                    }
                    // ESTADO ROJO: Error (no encontrado, día no laborable, etc.)
                    else if (response.es_dia_calendario) {
                        mostrarResultado(
                            'error',
                            '✖ ' + (response.calendario_info?.tipo?.replace('_', ' ') || 'Día No Laborable'),
                            `<small>${response.calendario_info?.nombre || 'No se registran asistencias'}</small>`
                        );
                        playErrorBeep();

                        if (html5QrcodeScanner) {
                            html5QrcodeScanner.pause();
                            setTimeout(() => {
                                if (isScanning) html5QrcodeScanner.resume();
                            }, 2000);
                        }
                    } else {
                        mostrarResultado(
                            'error',
                            '✖ Error',
                            `<small>${response.mensaje || 'Estudiante no encontrado'}</small>`
                        );
                        playErrorBeep();

                        if (html5QrcodeScanner) {
                            html5QrcodeScanner.pause();
                            setTimeout(() => {
                                if (isScanning) html5QrcodeScanner.resume();
                            }, 1500);
                        }
                    }
                }
            }, 'json').fail(function() {
                mostrarResultado('error', '✖ Error', '<small>Error del servidor</small>');
                playErrorBeep();
            });
        }

        function onScanFailure(error) {
            // Silenciar errores de escaneo continuo
        }

        function mostrarResultado(tipo, titulo, contenido, estudiante = null) {
            const card = $('#resultCard');
            card.removeClass('show');
            card.removeClass('success warning error');

            const iconos = {
                'success': 'check-circle-fill text-success',
                'warning': 'exclamation-triangle-fill text-warning',
                'error': 'x-circle-fill text-danger'
            };

            $('#resultIcon').html(`<i class="bi bi-${iconos[tipo]}"></i>`);

            setTimeout(() => {
                card.addClass(tipo + ' show');
                $('#resultTitle').text(titulo);
                $('#resultTime').text(new Date().toLocaleTimeString());
                $('#resultContent').html(contenido);
            }, 100);
        }

        function agregarARecientes(estudiante) {
            const timestamp = new Date().toLocaleTimeString();
            recientes.unshift({...estudiante, timestamp});

            if (recientes.length > 10) recientes.pop();

            let html = '';
            recientes.forEach(reg => {
                const esTarde = reg.estado === 'tarde';
                const badgeClass = esTarde ? 'warning' : 'success';
                const icon = esTarde ? 'clock-fill' : 'check-circle-fill';

                html += `
                    <div class="recent-item d-flex align-items-center py-2 border-bottom">
                        <div class="me-3">
                            <span class="badge bg-${badgeClass} estado-badge">
                                <i class="bi bi-${icon} me-1"></i>${reg.estado_abreviatura}
                            </span>
                        </div>
                        <div class="flex-grow-1">
                            <div class="fw-bold">${reg.nombre}</div>
                            <div class="small text-muted">${reg.grado} - "${reg.seccion}" • ${timestamp}</div>
                        </div>
                    </div>
                `;
            });

            $('#recentList').html(html);
        }

        function playBeep() {
            const audioContext = new (window.AudioContext || window.webkitAudioContext)();
            const oscillator = audioContext.createOscillator();
            const gainNode = audioContext.createGain();

            oscillator.connect(gainNode);
            gainNode.connect(audioContext.destination);

            oscillator.frequency.value = 800;
            oscillator.type = 'sine';

            gainNode.gain.setValueAtTime(0.3, audioContext.currentTime);
            gainNode.gain.exponentialRampToValueAtTime(0.01, audioContext.currentTime + 0.1);

            oscillator.start(audioContext.currentTime);
            oscillator.stop(audioContext.currentTime + 0.1);
        }

        function playErrorBeep() {
            const audioContext = new (window.AudioContext || window.webkitAudioContext)();
            const oscillator = audioContext.createOscillator();
            const gainNode = audioContext.createGain();

            oscillator.connect(gainNode);
            gainNode.connect(audioContext.destination);

            oscillator.frequency.value = 200;
            oscillator.type = 'sawtooth';

            gainNode.gain.setValueAtTime(0.3, audioContext.currentTime);
            gainNode.gain.exponentialRampToValueAtTime(0.01, audioContext.currentTime + 0.2);

            oscillator.start(audioContext.currentTime);
            oscillator.stop(audioContext.currentTime + 0.2);
        }

        function playWarningBeep() {
            const audioContext = new (window.AudioContext || window.webkitAudioContext)();
            const oscillator = audioContext.createOscillator();
            const gainNode = audioContext.createGain();

            oscillator.connect(gainNode);
            gainNode.connect(audioContext.destination);

            oscillator.frequency.value = 400;
            oscillator.type = 'triangle';

            gainNode.gain.setValueAtTime(0.3, audioContext.currentTime);
            gainNode.gain.exponentialRampToValueAtTime(0.01, audioContext.currentTime + 0.15);

            oscillator.start(audioContext.currentTime);
            oscillator.stop(audioContext.currentTime + 0.15);
        }

        // Ejecutar verificaciones al cargar la página
        $(document).ready(function() {
            verificarPermisosCamara();

            // Mostrar información del estado de HTTPS
            if (location.protocol !== 'https:' && location.hostname !== 'localhost' && location.hostname !== '127.0.0.1') {
                $('#resultCard').addClass('show error');
                $('#resultIcon').html('<i class="bi bi-exclamation-triangle-fill text-danger"></i>');
                $('#resultTitle').text('⚠ Advertencia de HTTPS');
                $('#resultTime').text(new Date().toLocaleTimeString());
                $('#resultContent').html(`
                    <strong>La cámara no funcionará sin HTTPS</strong><br><br>
                    Los navegadores modernos requieren una conexión segura (HTTPS) para acceder a la cámara.<br><br>
                    <strong>Soluciones:</strong><br>
                    1. Use localhost (si está en desarrollo)<br>
                    2. Configure un certificado SSL en XAMPP<br>
                    3. Use un servidor con HTTPS
                `);
            }
        });

        // Limpiar al salir de la página
        window.addEventListener('beforeunload', function() {
            if (html5QrcodeScanner && isScanning) {
                html5QrcodeScanner.stop();
            }
        });
    </script>
</body>
</html>
