<?php
require_once '../config/db.php';
require_once '../config/functions.php';

requerirRol(['admin', 'auxiliar', 'docente']);

$anio_seleccionado_id = getAnioLectivoSeleccionado();

if (!$anio_seleccionado_id) {
    die('Error: No hay año lectivo seleccionado. Configure un año lectivo activo primero.');
}

$anio_seleccionado = query("SELECT id, anio FROM anios_lectivos WHERE id = ?", [$anio_seleccionado_id])->fetch();

if (!$anio_seleccionado) {
    die('Error: Año lectivo seleccionado no encontrado. Vuelva a seleccionar un año lectivo.');
}

// Obtener estados de asistencia
$estados_asistencia = query("SELECT * FROM estados_asistencia WHERE estado = 'activo' ORDER BY nombre")->fetchAll();

// Por seguridad, verificar que anio_seleccionado sea un array
if (!is_array($anio_seleccionado) || !isset($anio_seleccionado['anio'])) {
    die('Error: Estructura de año lectivo inválida.');
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>SICA - Registro Manual</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css" rel="stylesheet">
    <link href="../assets/css/sica.css?v=5.1" rel="stylesheet">
    <style>
        .form-card {
            max-width: 700px;
            margin: 0 auto;
            background: white;
            border-radius: 20px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.1);
            padding: 40px;
        }
        .dni-input {
            font-size: 1.8rem;
            letter-spacing: 4px;
            text-align: center;
            font-weight: 600;
        }
        .estudiante-info {
            display: none;
            background: linear-gradient(135deg, #f0f9ff 0%, #e0f2fe 100%);
            border-radius: 15px;
            padding: 20px;
            margin-bottom: 20px;
        }
        .estudiante-info.show {
            display: block;
            animation: slideIn 0.3s ease-out;
        }
        @keyframes slideIn {
            from {
                opacity: 0;
                transform: translateY(-10px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        .info-row {
            display: flex;
            justify-content: space-between;
            padding: 8px 0;
            border-bottom: 1px solid rgba(0,0,0,0.05);
        }
        .info-row:last-child {
            border-bottom: none;
        }
        .info-label {
            font-weight: 600;
            color: #64748b;
        }
        .info-value {
            color: #1e293b;
            font-weight: 500;
        }
        .result-card {
            border-radius: 15px;
            padding: 25px;
            animation: slideIn 0.3s ease-out;
            display: none;
        }
        .result-card.show {
            display: block;
        }
        .result-card.success {
            background: linear-gradient(135deg, #d1fae5 0%, #a7f3d0 100%);
            border-left: 5px solid #10b981;
        }
        .result-card.error {
            background: linear-gradient(135deg, #fee2e2 0%, #fecaca 100%);
            border-left: 5px solid #ef4444;
        }
        .btn-sm {
            padding: 0.25rem 0.5rem;
            font-size: 0.75rem;
        }
        #modalEditar .modal-dialog {
            max-width: 400px;
        }
        #modalEditar .modal-body {
            padding: 1rem;
        }
        #modalEditar .modal-footer {
            padding: 0.75rem 1rem;
        }
    </style>
</head>
<body>
    <?php include '../includes/layout_elems.php'; ?>

    <div class="d-flex justify-content-between align-items-center mb-4">
        <div class="d-flex align-items-center">
            <button class="btn btn-link d-lg-none me-3" onclick="toggleSidebarMobile()">
                <i class="bi bi-list fs-4"></i>
            </button>
            <div>
                <h3 class="mb-1">Registro Manual de Asistencia</h3>
                <p class="text-muted mb-0">Ingrese el DNI del estudiante - <?= htmlspecialchars(is_array($anio_seleccionado) && isset($anio_seleccionado['anio']) ? $anio_seleccionado['anio'] : 'N/A') ?></p>
            </div>
        </div>
        <div>
            <a href="escanear_qr.php" class="btn btn-primary me-2">
                <i class="bi bi-qr-code-scan me-2"></i>Escáner QR
            </a>
            <a href="../dashboard.php" class="btn btn-outline-primary">
                <i class="bi bi-house-fill me-2"></i>Dashboard
            </a>
        </div>
    </div>

    <div class="form-card">
        <!-- Resultado del registro -->
        <div id="resultCard" class="result-card mb-4"></div>

        <!-- Información del estudiante (se muestra al buscar) -->
        <div id="estudianteInfo" class="estudiante-info">
            <h6 class="mb-3"><i class="bi bi-person-fill me-2"></i>Información del Estudiante</h6>
            <div class="info-row">
                <span class="info-label">DNI:</span>
                <span class="info-value" id="infoDni">-</span>
            </div>
            <div class="info-row">
                <span class="info-label">Nivel:</span>
                <span class="info-value" id="infoNivel">-</span>
            </div>
            <div class="info-row">
                <span class="info-label">Grado:</span>
                <span class="info-value" id="infoGrado">-</span>
            </div>
            <div class="info-row">
                <span class="info-label">Sección:</span>
                <span class="info-value" id="infoSeccion">-</span>
            </div>
            <div class="info-row">
                <span class="info-label">Nombre:</span>
                <span class="info-value" id="infoNombre">-</span>
            </div>
            <div class="info-row">
                <span class="info-label">Turno:</span>
                <span class="info-value" id="infoTurno">-</span>
            </div>
        </div>

        <form method="POST" id="formularioAsistencia">
            <div class="mb-4">
                <label class="form-label fw-bold">DNI del Estudiante</label>
                <div class="input-group input-group-lg">
                    <input type="text"
                           class="form-control dni-input"
                           name="dni"
                           id="dniInput"
                           maxlength="8"
                           placeholder="00000000"
                           pattern="\d{8}"
                           required
                           autofocus
                           value=""
                           oninput="this.value = this.value.replace(/\D/g, '')">
                    <button class="btn btn-primary" type="button" onclick="buscarEstudianteManual()">
                        <i class="bi bi-search"></i> Buscar
                    </button>
                </div>
                <div class="form-text text-center mt-3">Ingrese los 8 dígitos del DNI y haga clic en Buscar</div>
            </div>

            <div id="camposAdicionales" style="display: none;">
                <div class="mb-3">
                    <label class="form-label fw-bold">Estado de Asistencia</label>
                    <select class="form-select form-select-lg" name="id_estado_asistencia" id="estadoAsistencia">
                        <option value="">Seleccione estado...</option>
                        <?php foreach ($estados_asistencia as $estado): ?>
                            <option value="<?= $estado['id'] ?>"
                                    data-color="<?= $estado['color'] ?>"
                                    data-abbr="<?= $estado['abreviatura'] ?>">
                                <?= htmlspecialchars($estado['nombre']) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="row mb-3">
                    <div class="col-md-6">
                        <label class="form-label fw-bold">Fecha de Registro</label>
                        <input type="date"
                               class="form-control"
                               name="fecha_registro"
                               id="fechaRegistro"
                               value="<?= date('Y-m-d') ?>">
                    </div>
                    <div class="col-md-6">
                        <label class="form-label fw-bold">Hora de Registro</label>
                        <input type="time"
                               class="form-control"
                               name="hora_registro"
                               id="horaRegistro"
                               value="<?= date('H:i') ?>">
                    </div>
                </div>
                <div class="mb-3">
                    <label class="form-label fw-bold">Observaciones</label>
                    <textarea class="form-control"
                              name="observaciones"
                              id="observaciones"
                              rows="2"
                              placeholder="Opcional"></textarea>
                </div>

                <div class="d-grid gap-2">
                    <button type="submit" class="btn btn-lg" style="background: linear-gradient(135deg, var(--primary-color), #3b82f6); color: white; font-weight: 600;">
                        <i class="bi bi-check-circle-fill me-2"></i>Registrar Asistencia
                    </button>
                    <button type="button" class="btn btn-outline-secondary" onclick="limpiarFormulario()">
                        <i class="bi bi-x-circle me-2"></i>Cancelar / Nuevo
                    </button>
                </div>
            </div>
        </form>

        <!-- Historial de Asistencias -->
        <div id="historialSection" class="mt-5" style="display: none;">
            <div class="d-flex justify-content-between align-items-center mb-3">
                <h6 class="mb-0"><i class="bi bi-clock-history me-2"></i>Últimas 10 Asistencias</h6>
                <span class="badge bg-secondary" id="totalHistorial">0 registros</span>
            </div>
            <div class="table-responsive">
                <table class="table table-hover">
                    <thead>
                        <tr>
                            <th>Fecha</th>
                            <th>Hora</th>
                            <th>Estado</th>
                            <th>Tardanza</th>
                            <th>Método</th>
                            <th>Registrado Por</th>
                            <th class="text-end">Acciones</th>
                        </tr>
                    </thead>
                    <tbody id="historialBody">
                        <tr>
                            <td colspan="7" class="text-center text-muted">
                                <i class="bi bi-inbox fs-4 d-block mb-2"></i>
                                No hay asistencias registradas
                            </td>
                        </tr>
                    </tbody>
                </table>
            </div>
        </div>

        <div class="text-center mt-4">
            <a href="escanear_qr.php" class="text-decoration-none">
                <i class="bi bi-qr-code-scan me-2"></i>Usar escáner de código QR
            </a>
        </div>
    </div>

    <!-- Modal Editar Asistencia -->
    <div class="modal fade" id="modalEditar" tabindex="-1">
        <div class="modal-dialog modal-sm">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title"><i class="bi bi-pencil-fill me-2"></i>Editar Asistencia</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form id="formEditar">
                    <div class="modal-body">
                        <input type="hidden" id="editId">

                        <div class="mb-3">
                            <label class="form-label fw-bold">Fecha</label>
                            <input type="date" class="form-control" id="editFecha" required>
                        </div>

                        <div class="mb-3">
                            <label class="form-label fw-bold">Hora</label>
                            <input type="time" class="form-control" id="editHora" required>
                        </div>

                        <div class="mb-3">
                            <label class="form-label fw-bold">Estado de Asistencia</label>
                            <select class="form-select" id="editEstado" required>
                                <option value="">Seleccione...</option>
                                <?php foreach ($estados_asistencia as $estado): ?>
                                    <option value="<?= $estado['id'] ?>">
                                        <?= htmlspecialchars($estado['nombre']) ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>

                        <div class="mb-3">
                            <label class="form-label fw-bold">Observaciones</label>
                            <textarea class="form-control" id="editObservaciones" rows="3"></textarea>
                        </div>

                        <div class="alert alert-info small">
                            <i class="bi bi-info-circle me-2"></i>
                            Los minutos de tardanza se recalcularán automáticamente según la hora de ingreso del turno.
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                        <button type="submit" class="btn btn-primary">
                            <i class="bi bi-check-circle me-2"></i>Guardar Cambios
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <?php include '../includes/layout_scripts.php'; ?>

    <!-- jQuery (requerido para AJAX) -->
    <script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>

    <script>
        let estudianteEncontrado = null;
        let buscando = false;

        document.addEventListener('DOMContentLoaded', function() {
            const dniInput = document.getElementById('dniInput');
            dniInput.focus();

            dniInput.addEventListener('keypress', function(e) {
                if (e.key === 'Enter') {
                    e.preventDefault();
                    if (this.value.length === 8) {
                        buscarEstudianteManual();
                    } else {
                        mostrarResultado('error', '✖ Error', 'El DNI debe tener 8 dígitos');
                    }
                }
            });
        });

        function buscarEstudianteManual() {
            const dni = document.getElementById('dniInput').value;
            console.log('Buscando estudiante con DNI:', dni);

            if (dni.length !== 8) {
                mostrarResultado('error', '✖ Error', 'El DNI debe tener 8 dígitos');
                return;
            }

            console.log('DNI válido, iniciando búsqueda...');
            buscarEstudiante(dni);
        }

        function buscarEstudiante(dni) {
            console.log('Función buscarEstudiante llamada con DNI:', dni);

            if (dni.length !== 8) {
                ocultarInfoEstudiante();
                document.getElementById('camposAdicionales').style.display = 'none';
                document.getElementById('historialSection').style.display = 'none';
                return;
            }

            if (buscando) {
                console.log('Ya hay una búsqueda en progreso');
                return;
            }

            buscando = true;
            console.log('Iniciando petición AJAX...');

            $.get('../config/ajax.php', {
                action: 'buscar_estudiante',
                dni: dni
            }, function(response) {
                console.log('Respuesta recibida:', response);

                if (response.success) {
                    estudianteEncontrado = response.estudiante;
                    mostrarInfoEstudiante(response.estudiante);
                    document.getElementById('camposAdicionales').style.display = 'block';
                    // Cargar historial
                    cargarHistorial(dni);
                } else {
                    estudianteEncontrado = null;
                    ocultarInfoEstudiante();
                    document.getElementById('camposAdicionales').style.display = 'none';
                    document.getElementById('historialSection').style.display = 'none';
                    mostrarResultado('error', '✖ No encontrado', response.mensaje || 'Estudiante no encontrado');
                }
            }, 'json').fail(function(xhr, status, error) {
                console.error('Error en petición AJAX:', error);
                mostrarResultado('error', '✖ Error', 'Error de comunicación: ' + error);
            }).always(function() {
                buscando = false;
            });
        }

        function cargarHistorial(dni) {
            console.log('Cargando historial para DNI:', dni);

            $.get('ajax.php', {
                action: 'obtener_historial',
                dni: dni,
                limite: 10
            }, function(response) {
                console.log('Historial recibido:', response);

                if (response.success && response.asistencias.length > 0) {
                    mostrarHistorial(response.asistencias);
                } else {
                    mostrarHistorialVacio();
                }
            }, 'json').fail(function(xhr, status, error) {
                console.error('Error al cargar historial:', error);
                mostrarHistorialVacio();
            });
        }

        function mostrarHistorial(asistencias) {
            const tbody = document.getElementById('historialBody');
            let html = '';

            asistencias.forEach(function(asist) {
                // Badge según estado
                const badgeClass = getBadgeClass(asist.estado_codigo);
                const tardanza = asist.minutos_tardanza > 0
                    ? `<span class="badge bg-warning text-dark">${asist.minutos_tardanza} min</span>`
                    : '<span class="text-muted">-</span>';

                // Icono método
                const metodoIcon = asist.metodo_registro === 'qr'
                    ? '<i class="bi bi-qr-code-scan text-primary" title="QR"></i>'
                    : '<i class="bi bi-pencil-fill text-secondary" title="Manual"></i>';

                html += `
                    <tr>
                        <td>${formatearFecha(asist.fecha)}</td>
                        <td>${asist.hora_registro}</td>
                        <td><span class="badge ${badgeClass}">${asist.estado_nombre}</span></td>
                        <td>${tardanza}</td>
                        <td>${metodoIcon}</td>
                        <td><small>${asist.registrado_por}</small></td>
                        <td class="text-end">
                            <div class="btn-group" role="group">
                                <button class="btn btn-sm btn-outline-primary" onclick='editarAsistencia(${JSON.stringify(asist)})' title="Editar">
                                    <i class="bi bi-pencil-fill"></i>
                                </button>
                                <button class="btn btn-sm btn-outline-danger" onclick="eliminarAsistencia(${asist.id}, '${asist.fecha}', '${asist.estado_nombre}')" title="Eliminar">
                                    <i class="bi bi-trash-fill"></i>
                                </button>
                            </div>
                        </td>
                    </tr>
                `;
            });

            tbody.innerHTML = html;
            document.getElementById('totalHistorial').textContent = `${asistencias.length} registros`;
            document.getElementById('historialSection').style.display = 'block';
        }

        function mostrarHistorialVacio() {
            document.getElementById('historialBody').innerHTML = `
                <tr>
                    <td colspan="7" class="text-center text-muted">
                        <i class="bi bi-inbox fs-4 d-block mb-2"></i>
                        No hay asistencias registradas
                    </td>
                </tr>
            `;
            document.getElementById('totalHistorial').textContent = '0 registros';
            document.getElementById('historialSection').style.display = 'block';
        }

        function getBadgeClass(estadoCodigo) {
            const clases = {
                'ASI': 'bg-success',
                'TAR': 'bg-warning text-dark',
                'TJU': 'bg-info text-dark',
                'FJU': 'bg-primary',
                'FIN': 'bg-danger'
            };
            return clases[estadoCodigo] || 'bg-secondary';
        }

        function formatearFecha(fechaISO) {
            const fecha = new Date(fechaISO + 'T00:00:00');
            const options = { day: '2-digit', month: '2-digit', year: 'numeric' };
            return fecha.toLocaleDateString('es-PE', options);
        }

        function editarAsistencia(asistencia) {
            console.log('Editando asistencia:', asistencia);

            document.getElementById('editId').value = asistencia.id;
            document.getElementById('editFecha').value = asistencia.fecha;
            document.getElementById('editHora').value = asistencia.hora_registro;
            document.getElementById('editEstado').value = asistencia.id_estado_asistencia;
            document.getElementById('editObservaciones').value = asistencia.observaciones || '';

            const modal = new bootstrap.Modal(document.getElementById('modalEditar'));
            modal.show();
        }

        function eliminarAsistencia(id, fecha, estado) {
            if (!confirm(`¿Eliminar el registro del ${formatearFecha(fecha)} (${estado})?`)) {
                return;
            }

            $.post('ajax.php', {
                action: 'eliminar_asistencia',
                id: id
            }, function(response) {
                if (response.success) {
                    mostrarResultado('success', '✓ Eliminado', 'Registro eliminado correctamente');
                    // Recargar historial
                    if (estudianteEncontrado) {
                        cargarHistorial(estudianteEncontrado.dni);
                    }
                } else {
                    mostrarResultado('error', '✖ Error', response.mensaje || 'Error al eliminar');
                }
            }, 'json').fail(function() {
                mostrarResultado('error', '✖ Error', 'Error de comunicación con el servidor');
            });
        }

        // Formulario de edición
        document.getElementById('formEditar').addEventListener('submit', function(e) {
            e.preventDefault();

            const formData = {
                action: 'editar_asistencia',
                id: document.getElementById('editId').value,
                fecha: document.getElementById('editFecha').value,
                hora: document.getElementById('editHora').value,
                id_estado: document.getElementById('editEstado').value,
                observaciones: document.getElementById('editObservaciones').value
            };

            $.post('ajax.php', formData, function(response) {
                if (response.success) {
                    // Cerrar modal
                    const modal = bootstrap.Modal.getInstance(document.getElementById('modalEditar'));
                    modal.hide();

                    mostrarResultado('success', '✓ Actualizado', 'Asistencia actualizada correctamente');

                    // Recargar historial
                    if (estudianteEncontrado) {
                        cargarHistorial(estudianteEncontrado.dni);
                    }
                } else {
                    alert('Error: ' + (response.mensaje || 'Error al actualizar'));
                }
            }, 'json').fail(function() {
                alert('Error de comunicación con el servidor');
            });
        });

        function mostrarInfoEstudiante(est) {
            document.getElementById('infoDni').textContent = est.dni;
            document.getElementById('infoNivel').textContent = est.nivel;
            document.getElementById('infoGrado').textContent = est.grado;
            document.getElementById('infoSeccion').textContent = '"' + est.seccion + '"';
            document.getElementById('infoNombre').textContent = est.apellido_paterno + ' ' + est.apellido_materno + ', ' + est.nombres;
            document.getElementById('infoTurno').textContent = est.turno;
            document.getElementById('estudianteInfo').classList.add('show');
        }

        function ocultarInfoEstudiante() {
            document.getElementById('estudianteInfo').classList.remove('show');
        }

        document.getElementById('formularioAsistencia').addEventListener('submit', function(e) {
            e.preventDefault();

            // Validaciones manuales
            if (!estudianteEncontrado) {
                mostrarResultado('error', '✖ Error', 'Debe buscar un estudiante primero');
                return;
            }

            const estadoAsistencia = document.getElementById('estadoAsistencia').value;
            if (!estadoAsistencia) {
                mostrarResultado('error', '✖ Error', 'Debe seleccionar un estado de asistencia');
                document.getElementById('estadoAsistencia').focus();
                return;
            }

            const fechaRegistro = document.getElementById('fechaRegistro').value;
            if (!fechaRegistro) {
                mostrarResultado('error', '✖ Error', 'Debe seleccionar una fecha de registro');
                document.getElementById('fechaRegistro').focus();
                return;
            }

            const horaRegistro = document.getElementById('horaRegistro').value;
            if (!horaRegistro) {
                mostrarResultado('error', '✖ Error', 'Debe seleccionar una hora de registro');
                document.getElementById('horaRegistro').focus();
                return;
            }

            const formData = new FormData(this);
            formData.append('action', 'registrar_asistencia_manual');

            $.ajax({
                url: '../config/ajax.php',
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                success: function(response) {
                    if (response.success) {
                        mostrarResultado('success', '✓ Registrado', `
                            <strong>${response.estudiante.nombre}</strong><br>
                            <small class="text-muted">DNI: ${response.estudiante.dni}</small><br>
                            <small class="text-muted">${response.estudiante.nivel} - ${response.estudiante.grado} "${response.estudiante.seccion}"</small><br>
                            <small class="text-muted">Fecha: ${response.estudiante.fecha}</small><br>
                            <small class="text-muted">Hora: ${response.estudiante.hora}</small><br>
                            <small class="text-muted">Estado: ${response.estudiante.estado_nombre}</small>
                        `);

                        setTimeout(function() {
                            limpiarFormulario();
                        }, 3000);
                    } else {
                        // Manejar respuesta de día calendario (feriado, vacación, etc.)
                        if (response.es_dia_calendario && response.calendario_info) {
                            const tipoLabel = {
                                'feriado': 'Feriado',
                                'vacacion': 'Vacación',
                                'dia_no_laboral': 'Día No Laboral',
                                'fin_semana_largo': 'Fin de Semana Largo'
                            }[response.calendario_info.tipo] || response.calendario_info.tipo;

                            const iconos = {
                                'feriado': 'calendar-x',
                                'vacacion': 'calendar-week',
                                'dia_no_laboral': 'calendar3',
                                'fin_semana_largo': 'calendar-range'
                            }[response.calendario_info.tipo] || 'calendar-x';

                            const colores = {
                                'feriado': 'text-danger',
                                'vacacion': 'text-primary',
                                'dia_no_laboral': 'text-warning',
                                'fin_semana_largo': 'text-info'
                            }[response.calendario_info.tipo] || 'text-danger';

                            mostrarResultado('error', `Día No Laborable: ${tipoLabel}`, `
                                <div class="d-flex align-items-start">
                                    <div class="me-3">
                                        <i class="bi bi-${iconos} ${colores} fs-2"></i>
                                    </div>
                                    <div>
                                        <p class="mb-1"><strong>${response.calendario_info.nombre}</strong></p>
                                        <small class="text-muted">${response.mensaje}</small>
                                    </div>
                                </div>
                            `);
                        } else {
                            mostrarResultado('error', '✖ Error', response.mensaje);
                        }
                    }
                },
                error: function() {
                    mostrarResultado('error', '✖ Error', 'Error de comunicación con el servidor');
                }
            });
        });

        function mostrarResultado(tipo, titulo, mensaje) {
            const card = document.getElementById('resultCard');
            card.className = 'result-card ' + tipo + ' show';
            card.innerHTML = `
                <div class="d-flex align-items-center mb-2">
                    <div class="me-3 fs-2">
                        <i class="bi bi-${tipo === 'success' ? 'check-circle-fill text-success' : 'x-circle-fill text-danger'}"></i>
                    </div>
                    <div>
                        <h5 class="mb-0">${titulo}</h5>
                        <p class="text-muted small mb-0">${new Date().toLocaleTimeString()}</p>
                    </div>
                </div>
                <hr>
                ${mensaje}
            `;
        }

        function limpiarFormulario() {
            document.getElementById('formularioAsistencia').reset();
            document.getElementById('dniInput').value = '';
            document.getElementById('fechaRegistro').value = new Date().toISOString().slice(0,10);
            document.getElementById('horaRegistro').value = new Date().toTimeString().slice(0,5);
            estudianteEncontrado = null;
            ocultarInfoEstudiante();
            document.getElementById('camposAdicionales').style.display = 'none';
            document.getElementById('resultCard').classList.remove('show');
            document.getElementById('dniInput').focus();
        }
    </script>
</body>
</html>
