# GENERACIÓN DE FERIADOS POR AÑO LECTIVO

## ¿Cómo Funciona?

El sistema SICA genera **feriados por año lectivo**. Esto significa:

- **Cada año lectivo tiene sus propios feriados** con fechas calculadas correctamente
- Los feriados están **asignados al año lectivo** (no son globales)
- El sistema **filtra automáticamente** los feriados según el año lectivo activo

## Ejemplo Práctico:

```
Año lectivo activo: 2026 (ID = 2)

Resultado en pantalla:
- Feriados del año lectivo 2026: ✓ Se muestran
- Feriados del año lectivo 2025: ✗ No se muestran
```

---

## CÓMO GENERAR FERIADOS

### Script PHP

**Pasos:**

1. Abrir terminal/consola

2. Navegar a la carpeta config:
```bash
cd c:\xampp\htdocs\SICA\config
```

3. Ejecutar el script:
```bash
# Generar feriados para el año lectivo 2026
php generar_feriados.php 2026

# O para el año lectivo 2025
php generar_feriados.php 2025
```

4. Verificar resultado:
```bash
# Deberías ver:
# === GENERADOR DE FERIADOS PARA SICA ===
# Generando feriados del año 2026...
#
# ✓ Año lectivo encontrado: 2026 (ID: 2)
# ✓ Estado: activo
#
# 1. Eliminando feriados existentes del año lectivo 2026...
#    ✓ Feriados antiguos eliminados
#
#    ✓ 2026-01-01 - Año Nuevo
#    ✓ 2026-04-02 - Jueves Santo
#    ✓ 2026-04-03 - Viernes Santo
#    ...
```

---

## FERIADOS GENERADOS (13 por año)

| # | Feriado | Fecha | Tipo |
|---|---------|-------|------|
| 1 | Año Nuevo | 1 de enero | Fijo |
| 2 | Jueves Santo | Variable | Calculado |
| 3 | Viernes Santo | Variable | Calculado |
| 4 | Día del Trabajo | 1 de mayo | Fijo |
| 5 | San Pedro y San Pablo | 29 de junio | Fijo |
| 6 | Fiestas Patrias - Día 1 | 28 de julio | Fijo |
| 7 | Fiestas Patrias - Día 2 | 29 de julio | Fijo |
| 8 | Santa Rosa de Lima | 30 de agosto | Fijo |
| 9 | Combate de Angamos | 8 de octubre | Fijo |
| 10 | Todos los Santos | 1 de noviembre | Fijo |
| 11 | Inmaculada Concepción | 8 de diciembre | Fijo |
| 12 | Navidad | 25 de diciembre | Fijo |

---

## ALGORITMO DE SEMANA SANTA

El script usa el **Algoritmo de Gauss** para calcular las fechas de Semana Santa:

```
Domingo de Pascua = Calculado con fórmula astronómica
Jueves Santo     = Pascua - 3 días
Viernes Santo    = Pascua - 2 días
```

**Ejemplo de resultados correctos:**
- 2025: Jueves 10 de abril, Viernes 11 de abril
- 2026: Jueves 2 de abril, Viernes 3 de abril
- 2027: Jueves 22 de abril, Viernes 23 de abril

---

## FILTRADO POR AÑO LECTIVO

El sistema filtra automáticamente los feriados según el año lectivo activo:

**Consulta SQL:**
```sql
SELECT *
FROM calendario
WHERE estado = 'activo'
  AND (
      aniolectivo_id = ?  -- Del año lectivo activo
      OR YEAR(fecha) = ?  -- O del año correspondiente
  )
ORDER BY fecha ASC
```

**Resultado Práctico:**

```
Año lectivo activo: 2026 (ID = 2)

Pantalla muestra:
✓ 2026-01-01 - Año Nuevo
✓ 2026-04-02 - Jueves Santo
✓ 2026-04-03 - Viernes Santo
✓ 2026-07-28 - Fiestas Patrias - Día 1
... (todos los feriados del 2026)

NO muestra:
✗ Cualquier feriado de otros años lectivos
```

---

## MANTENIMIENTO

### ¿Cuándo Generar Nuevos Feriados?

**Recomendación:** Generar feriados cuando **crees un nuevo año lectivo**.

```bash
# Ejemplo: Creaste el año lectivo 2027
# Generar sus feriados:
php generar_feriados.php 2027
```

### ¿Qué Hacer si Cambia el Año Lectivo Activo?

**NADA.** El filtro es automático:

1. Cambias el año lectivo activo en la configuración
2. El sistema muestra automáticamente los feriados de ese año
3. No requiere intervención manual

---

## VERIFICACIÓN

### Consulta SQL para Verificar Feriados por Año Lectivo:

```sql
SELECT
    an.anio as año_lectivo,
    COUNT(c.id) as total_feriados
FROM calendario c
RIGHT JOIN anios_lectivos an ON c.aniolectivo_id = an.id
GROUP BY an.id, an.anio
ORDER BY an.anio;
```

**Resultado esperado:**
| año_lectivo | total_feriados |
|-------------|----------------|
| 2025 | 13 |
| 2026 | 13 |
| 2027 | 0 |
| 2028 | 0 |

---

## TROUBLESHOOTING

### Problema: "No se muestran los feriados del año seleccionado"

**Solución:**
1. Verificar que los feriados existen en la base de datos:
```sql
SELECT * FROM calendario
WHERE aniolectivo_id = (SELECT id FROM anios_lectivos WHERE anio LIKE '%2026%')
ORDER BY fecha;
```

2. Si no aparecen, ejecutar el script generador:
```bash
php generar_feriados.php 2026
```

### Problema: "Las fechas de Semana Santa están incorrectas"

**Solución:**
1. El script PHP calcula automáticamente las fechas correctas
2. Verifica el año que estás usando (ej: 2026 vs 2025)

### Problema: "Tabla aparece vacía"

**Causas posibles:**
1. **No se han generado feriados para ese año lectivo** → Ejecutar el script generador
2. **El año lectivo no tiene feriados asignados** → Ejecutar el script generador

---

## ESTRUCTURA DE DATOS

### Tabla calendario:

```sql
CREATE TABLE calendario (
    id INT AUTO_INCREMENT PRIMARY KEY,
    fecha DATE NOT NULL,
    tipo ENUM('feriado', 'vacacion', 'dia_no_laboral', 'fin_semana_largo'),
    nombre VARCHAR(100) NOT NULL,
    descripcion TEXT,
    aniolectivo_id INT NOT NULL,  -- ← Asignado al año lectivo específico
    afecta_todos_niveles BOOLEAN DEFAULT TRUE,
    niveles_afectados VARCHAR(255) NULL,
    estado ENUM('activo', 'inactivo') DEFAULT 'activo'
);
```

**Ejemplos:**

| fecha | tipo | nombre | aniolectivo_id | ¿Qué significa? |
|-------|------|--------|----------------|-----------------|
| 2026-07-28 | feriado | Fiestas Patrias | 2 | Feriado del año lectivo ID=2 (2026) |
| 2026-07-15 | vacacion | Vacaciones julio | 2 | Vacación del año lectivo ID=2 (2026) |
| 2025-07-28 | feriado | Fiestas Patrias | 1 | Feriado del año lectivo ID=1 (2025) |

---

## ARCHIVOS DEL SISTEMA

```
SICA/
├── config/
│   ├── calendario.php                  # Interfaz CRUD del calendario
│   ├── generar_feriados.php            # Script PHP (generador por año lectivo)
│   └── INSTRUCCIONES_FERIADOS.md       # Este archivo
├── database/
│   └── migracion_calendario.sql        # Migración inicial
└── config/ajax.php                     # Validación de asistencia con feriados
```

---

## RESUMEN

✅ **Cada año lectivo tiene sus propios feriados**
✅ **Semana Santa se calcula automáticamente**
✅ **El filtro es automático por año lectivo activo**
✅ **Se generan 13 feriados por año lectivo**
✅ **Puedes agregar feriados adicionales desde la interfaz**

**Para generar feriados de un año lectivo:**
```bash
cd c:\xampp\htdocs\SICA\config
php generar_feriados.php 2026
```
