<?php
require_once '../config/db.php';
require_once '../config/functions.php';

requerirRol(['admin']);

$anio_seleccionado = getAnioLectivoSeleccionado();

// Obtener niveles activos
$niveles = query("SELECT id, nombre as nivel FROM niveles WHERE estado = 'activo' ORDER BY nombre")->fetchAll();

// Obtener filtros
$filtro_nivel = $_GET['filtro_nivel'] ?? '';

// Construir consulta con filtros
$sql = "SELECT aa.*, n.nombre as nombre_nivel
        FROM areas_academicas aa
        INNER JOIN niveles n ON aa.id_nivel = n.id
        WHERE aa.id_anio_lectivo = :anio";
$params = ['anio' => $anio_seleccionado];

if ($filtro_nivel) {
    $sql .= " AND aa.id_nivel = :filtro_nivel";
    $params['filtro_nivel'] = $filtro_nivel;
}

$sql .= " ORDER BY n.nombre, aa.area";
$areas = query($sql, $params)->fetchAll();

$mensaje = '';
$tipo_mensaje = '';

// Procesar formulario
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $accion = $_POST['accion'] ?? '';
    $id = $_POST['id'] ?? '';
    $id_nivel = $_POST['id_nivel'] ?? '';
    $area = trim($_POST['area'] ?? '');
    $estado = $_POST['estado'] ?? 'activo';

    if ($accion === 'crear' || $accion === 'editar') {
        // Validaciones
        if (empty($id_nivel)) {
            $mensaje = 'Debe seleccionar un nivel educativo';
            $tipo_mensaje = 'danger';
        } elseif (empty($area)) {
            $mensaje = 'El nombre del área es obligatorio';
            $tipo_mensaje = 'danger';
        } else {
            try {
                if ($accion === 'crear') {
                    // Verificar duplicado
                    $check = query("SELECT COUNT(*) as total FROM areas_academicas
                                   WHERE id_nivel = :nivel AND id_anio_lectivo = :anio AND area = :area",
                                   ['nivel' => $id_nivel, 'anio' => $anio_seleccionado, 'area' => $area])->fetch();

                    if ($check['total'] > 0) {
                        $mensaje = 'Ya existe esta área académica para este nivel educativo';
                        $tipo_mensaje = 'danger';
                    } else {
                        $sql = "INSERT INTO areas_academicas (id_nivel, id_anio_lectivo, area, estado)
                                VALUES (:nivel, :anio, :area, :estado)";
                        query($sql, [
                            'nivel' => $id_nivel,
                            'anio' => $anio_seleccionado,
                            'area' => $area,
                            'estado' => $estado
                        ]);
                        $mensaje = 'Área académica creada exitosamente';
                        $tipo_mensaje = 'success';
                        // Recargar áreas
                        $areas = query("SELECT aa.*, n.nombre as nombre_nivel
                                       FROM areas_academicas aa
                                       INNER JOIN niveles n ON aa.id_nivel = n.id
                                       WHERE aa.id_anio_lectivo = :anio
                                       ORDER BY n.nombre, aa.area",
                                       ['anio' => $anio_seleccionado])->fetchAll();
                    }
                } elseif ($accion === 'editar') {
                    // Verificar duplicado (excluyendo el registro actual)
                    $check = query("SELECT COUNT(*) as total FROM areas_academicas
                                   WHERE id_nivel = :nivel AND id_anio_lectivo = :anio AND area = :area AND id != :id",
                                   ['nivel' => $id_nivel, 'anio' => $anio_seleccionado, 'area' => $area, 'id' => $id])->fetch();

                    if ($check['total'] > 0) {
                        $mensaje = 'Ya existe esta área académica para este nivel educativo';
                        $tipo_mensaje = 'danger';
                    } else {
                        $sql = "UPDATE areas_academicas
                                SET id_nivel = :nivel, area = :area, estado = :estado
                                WHERE id = :id";
                        query($sql, [
                            'nivel' => $id_nivel,
                            'area' => $area,
                            'estado' => $estado,
                            'id' => $id
                        ]);
                        $mensaje = 'Área académica actualizada exitosamente';
                        $tipo_mensaje = 'success';
                        // Recargar áreas
                        $areas = query("SELECT aa.*, n.nombre as nombre_nivel
                                       FROM areas_academicas aa
                                       INNER JOIN niveles n ON aa.id_nivel = n.id
                                       WHERE aa.id_anio_lectivo = :anio
                                       ORDER BY n.nombre, aa.area",
                                       ['anio' => $anio_seleccionado])->fetchAll();
                    }
                }
            } catch (PDOException $e) {
                $mensaje = 'Error al guardar: ' . $e->getMessage();
                $tipo_mensaje = 'danger';
            }
        }
    } elseif ($accion === 'eliminar') {
        try {
            query("DELETE FROM areas_academicas WHERE id = :id", ['id' => $id]);
            $mensaje = 'Área académica eliminada exitosamente';
            $tipo_mensaje = 'success';
            // Recargar áreas
            $areas = query("SELECT aa.*, n.nombre as nombre_nivel
                           FROM areas_academicas aa
                           INNER JOIN niveles n ON aa.id_nivel = n.id
                           WHERE aa.id_anio_lectivo = :anio
                           ORDER BY n.nombre, aa.area",
                           ['anio' => $anio_seleccionado])->fetchAll();
        } catch (PDOException $e) {
            $mensaje = 'Error al eliminar: ' . $e->getMessage();
            $tipo_mensaje = 'danger';
        }
    } elseif ($accion === 'cambiar_estado') {
        try {
            $nuevo_estado = $estado === 'activo' ? 'inactivo' : 'activo';
            query("UPDATE areas_academicas SET estado = :estado WHERE id = :id",
                   ['estado' => $nuevo_estado, 'id' => $id]);
            $mensaje = 'Estado actualizado exitosamente';
            $tipo_mensaje = 'success';
            // Recargar áreas
            $areas = query("SELECT aa.*, n.nombre as nombre_nivel
                           FROM areas_academicas aa
                           INNER JOIN niveles n ON aa.id_nivel = n.id
                           WHERE aa.id_anio_lectivo = :anio
                           ORDER BY n.nombre, aa.area",
                           ['anio' => $anio_seleccionado])->fetchAll();
        } catch (PDOException $e) {
            $mensaje = 'Error al cambiar estado: ' . $e->getMessage();
            $tipo_mensaje = 'danger';
        }
    }
}

// Obtener un área específica para edición (si se pasó ID por URL)
$area_editar = null;
if (isset($_GET['editar'])) {
    $id_editar = intval($_GET['editar']);
    $area_editar = query("SELECT * FROM areas_academicas WHERE id = :id", ['id' => $id_editar])->fetch();
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>SICA - Áreas Académicas</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css" rel="stylesheet">
    <link href="../assets/css/sica.css?v=5.1" rel="stylesheet">
</head>
<body>
    <nav class="sidebar">
        <?php include '../includes/sidebar.php'; ?>
    </nav>

    <div class="main-content">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <div>
                <h3 class="mb-1">Áreas Académicas</h3>
                <p class="text-muted mb-0">Gestione las áreas académicas por nivel educativo (<?= $info_anio['anio'] ?? '' ?>)</p>
            </div>
            <a href="index.php" class="btn btn-outline-secondary">
                <i class="bi bi-arrow-left"></i> Volver
            </a>
        </div>

        <?php if ($mensaje): ?>
            <div class="alert alert-<?= $tipo_mensaje ?> alert-dismissible fade show" role="alert">
                <?= htmlspecialchars($mensaje) ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <!-- Filtros -->
        <form method="GET" class="mb-4">
            <div class="row g-2">
                <div class="col-md-6">
                    <select name="filtro_nivel" class="form-select form-select-sm" onchange="this.form.submit()">
                        <option value="">Todos los niveles</option>
                        <?php foreach ($niveles as $n): ?>
                            <option value="<?= $n['id'] ?>" <?= $filtro_nivel == $n['id'] ? 'selected' : '' ?>>
                                <?= $n['nivel'] ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="col-md-6">
                    <?php if ($filtro_nivel): ?>
                        <a href="areas_academicas.php" class="btn btn-sm btn-outline-secondary w-100">
                            <i class="bi bi-x-lg me-1"></i>Limpiar Filtro
                        </a>
                    <?php endif; ?>
                </div>
            </div>
        </form>

        <div class="row">
            <!-- Formulario -->
            <div class="col-lg-4 mb-4">
                <div class="card">
                    <div class="card-header bg-primary text-white">
                        <h5 class="mb-0"><i class="bi bi-plus-circle"></i>
                            <?= $area_editar ? 'Editar Área Académica' : 'Nueva Área Académica' ?>
                        </h5>
                    </div>
                    <div class="card-body">
                        <form method="POST" id="formArea">
                            <input type="hidden" name="accion" value="<?= $area_editar ? 'editar' : 'crear' ?>">
                            <input type="hidden" name="id" value="<?= $area_editar['id'] ?? '' ?>">

                            <div class="mb-3">
                                <label for="id_nivel" class="form-label">Nivel Educativo <span class="text-danger">*</span></label>
                                <select class="form-select" id="id_nivel" name="id_nivel" required>
                                    <option value="">Seleccione nivel...</option>
                                    <?php foreach ($niveles as $nivel): ?>
                                        <option value="<?= $nivel['id'] ?>"
                                            <?= (isset($area_editar['id_nivel']) && $area_editar['id_nivel'] == $nivel['id']) ? 'selected' : '' ?>>
                                            <?= htmlspecialchars($nivel['nivel']) ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>

                            <div class="mb-3">
                                <label for="area" class="form-label">Área Académica <span class="text-danger">*</span></label>
                                <input type="text" class="form-control" id="area" name="area"
                                    value="<?= htmlspecialchars($area_editar['area'] ?? '') ?>"
                                    placeholder="Ej: Matemática, Comunicación, etc." required>
                            </div>

                            <div class="mb-3">
                                <label for="estado" class="form-label">Estado</label>
                                <select class="form-select" id="estado" name="estado">
                                    <option value="activo" <?= (isset($area_editar['estado']) && $area_editar['estado'] === 'activo') ? 'selected' : '' ?>>
                                        Activo
                                    </option>
                                    <option value="inactivo" <?= (isset($area_editar['estado']) && $area_editar['estado'] === 'inactivo') ? 'selected' : '' ?>>
                                        Inactivo
                                    </option>
                                </select>
                            </div>

                            <div class="d-flex gap-2">
                                <button type="submit" class="btn btn-primary flex-grow-1">
                                    <i class="bi bi-save"></i> <?= $area_editar ? 'Actualizar' : 'Guardar' ?>
                                </button>
                                <?php if ($area_editar): ?>
                                    <a href="areas_academicas.php" class="btn btn-secondary">
                                        <i class="bi bi-x-circle"></i> Cancelar
                                    </a>
                                <?php endif; ?>
                            </div>
                        </form>
                    </div>
                </div>
            </div>

            <!-- Listado -->
            <div class="col-lg-8">
                <div class="card">
                    <div class="card-header bg-white d-flex justify-content-between align-items-center">
                        <h5 class="mb-0"><i class="bi bi-list-ul"></i> Listado de Áreas Académicas</h5>
                        <span class="badge bg-primary"><?= count($areas) ?> áreas</span>
                    </div>
                    <div class="card-body p-0">
                        <?php if (empty($areas)): ?>
                            <div class="text-center py-5">
                                <i class="bi bi-inbox text-muted" style="font-size: 3rem;"></i>
                                <p class="text-muted mt-3">No hay áreas académicas registradas</p>
                                <p class="small text-muted">Registre áreas académicas usando el formulario</p>
                            </div>
                        <?php else: ?>
                            <div class="table-responsive">
                                <table class="table table-hover mb-0">
                                    <thead class="table-light">
                                        <tr>
                                            <th>Nivel</th>
                                            <th>Área Académica</th>
                                            <th>Estado</th>
                                            <th class="text-center">Acciones</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($areas as $area): ?>
                                            <tr>
                                                <td>
                                                    <span class="fw-bold"><?= htmlspecialchars($area['nombre_nivel']) ?></span>
                                                </td>
                                                <td><?= htmlspecialchars($area['area']) ?></td>
                                                <td>
                                                    <?php if ($area['estado'] === 'activo'): ?>
                                                        <span class="badge bg-success">
                                                            <i class="bi bi-check-circle"></i> Activo
                                                        </span>
                                                    <?php else: ?>
                                                        <span class="badge bg-secondary">
                                                            <i class="bi bi-dash-circle"></i> Inactivo
                                                        </span>
                                                    <?php endif; ?>
                                                </td>
                                                <td class="text-center">
                                                    <div class="btn-group" role="group">
                                                        <a href="?editar=<?= $area['id'] ?>"
                                                            class="btn btn-sm btn-outline-primary"
                                                            title="Editar">
                                                            <i class="bi bi-pencil"></i>
                                                        </a>
                                                        <button type="button"
                                                            class="btn btn-sm btn-outline-<?= $area['estado'] === 'activo' ? 'warning' : 'success' ?>"
                                                            onclick="cambiarEstado(<?= $area['id'] ?>, '<?= $area['estado'] ?>')"
                                                            title="<?= $area['estado'] === 'activo' ? 'Desactivar' : 'Activar' ?>">
                                                            <i class="bi bi-<?= $area['estado'] === 'activo' ? 'pause' : 'play' ?>-circle"></i>
                                                        </button>
                                                        <button type="button"
                                                            class="btn btn-sm btn-outline-danger"
                                                            onclick="eliminarArea(<?= $area['id'] ?>, '<?= htmlspecialchars($area['area']) ?>')"
                                                            title="Eliminar">
                                                            <i class="bi bi-trash"></i>
                                                        </button>
                                                    </div>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Información -->
                <div class="card mt-3">
                    <div class="card-body">
                        <h6 class="card-title"><i class="bi bi-info-circle text-primary"></i> Información</h6>
                        <p class="small text-muted mb-0">
                            Las <strong>Áreas Académicas</strong> permiten organizar las asignaturas o materias que se dictan en cada nivel educativo (Inicial, Primaria, Secundaria). Cada nivel puede tener diferentes áreas según su currículo.
                        </p>
                        <ul class="small text-muted mt-2">
                            <li>Ejemplo de Inicial: Comunicación, Matemática, Psicomotricidad</li>
                            <li>Ejemplo de Primaria: Matemática, Comunicación, Ciencia y Ambiente</li>
                            <li>Ejemplo de Secundaria: Matemática, Comunicación, Física, Química, Biología</li>
                        </ul>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
    <script>
    function cambiarEstado(id, estadoActual) {
        const nuevoEstado = estadoActual === 'activo' ? 'inactivo' : 'activo';
        const mensaje = nuevoEstado === 'activo'
            ? '¿Desea activar esta área académica?'
            : '¿Desea desactivar esta área académica?';

        if (confirm(mensaje)) {
            const form = document.createElement('form');
            form.method = 'POST';
            form.innerHTML = `
                <input type="hidden" name="accion" value="cambiar_estado">
                <input type="hidden" name="id" value="${id}">
                <input type="hidden" name="estado" value="${estadoActual}">
            `;
            document.body.appendChild(form);
            form.submit();
        }
    }

    function eliminarArea(id, nombre) {
        if (confirm(`¿Está seguro de eliminar el área académica "${nombre}"?\n\nEsta acción no se puede deshacer.`)) {
            const form = document.createElement('form');
            form.method = 'POST';
            form.innerHTML = `
                <input type="hidden" name="accion" value="eliminar">
                <input type="hidden" name="id" value="${id}">
            `;
            document.body.appendChild(form);
            form.submit();
        }
    }
    </script>
</body>
</html>
