<?php
/**
 * Script de restauración de base de datos SICA
 * Ejecutar desde: http://localhost/SICA/database/restaurar_completo.php
 */

set_time_limit(300); // 5 minutos

echo "<pre>";
echo "=== RESTAURANDO BASE DE DATOS SICA ===\n\n";

try {
    // Conectar sin seleccionar BD
    $dsn = "mysql:host=localhost;charset=utf8mb4";
    $options = [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::MYSQL_ATTR_USE_BUFFERED_QUERY => true,
    ];
    $pdo = new PDO($dsn, 'root', '', $options);

    // 1. Eliminar base de datos si existe
    echo "1. Eliminando base de datos anterior...\n";
    $pdo->exec("DROP DATABASE IF EXISTS sica2025");
    echo "   ✓ Eliminado\n\n";

    // 2. Crear nueva base de datos
    echo "2. Creando nueva base de datos...\n";
    $pdo->exec("CREATE DATABASE sica2025 CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
    $pdo->exec("USE sica2025");
    echo "   ✓ Creada y seleccionada\n\n";

    // 3. Leer scripts SQL
    echo "3. Leyendo scripts SQL...\n";
    $sql1 = file_get_contents(__DIR__ . '/sica_completo_restauracion.sql');
    $sql2 = file_get_contents(__DIR__ . '/datos_basicos_completos.sql');
    $sql_completo = $sql1 . "\n\n" . $sql2;
    echo "   ✓ Scripts leídos\n\n";

    // 4. Ejecutar consultas
    echo "4. Ejecutando consultas SQL...\n";

    // Importar usando mysqli multi_query (mejor para scripts SQL grandes)
    $mysqli = new mysqli('localhost', 'root', '', 'sica2025');
    if ($mysqli->connect_error) {
        throw new Exception("Error de conexión MySQLi: " . $mysqli->connect_error);
    }

    // Ejecutar el script completo
    $mysqli->multi_query($sql_completo);

    $ejecutadas = 0;
    $errores = 0;

    // Procesar todos los resultados
    do {
        if ($result = $mysqli->store_result()) {
            $result->free();
        }
        $ejecutadas++;

        if ($mysqli->errno && $errores < 10) {
            $errores++;
            echo "   ⚠ Error: " . $mysqli->error . "\n";
        }
    } while ($mysqli->more_results() && $mysqli->next_result());

    $mysqli->close();

    echo "   ✓ Total consultas ejecutadas: $ejecutadas\n";
    if ($errores > 0) {
        echo "   ⚠ Errores: $errores\n";
    }
    echo "\n";

    // 5. Verificación
    echo "5. Verificando datos insertados...\n";
    echo str_repeat("-", 50) . "\n";

    $tablas = [
        'anios_lectivos',
        'niveles',
        'turnos',
        'grados',
        'secciones',
        'horarios',
        'estados_asistencia',
        'tipos_incidencia',
        'areas_academicas',
        'calendario',
        'configuracion',
        'usuarios',
        'estudiantes',
        'docentes',
        'auxiliares',
        'apoderados',
        'estudiantes_apoderados'
    ];

    foreach ($tablas as $tabla) {
        try {
            $stmt = $pdo->query("SELECT COUNT(*) FROM $tabla");
            $count = $stmt->fetchColumn();
            $status = $count > 0 ? "✓" : "⚠";
            echo sprintf("%s %-30s %5d\n", $status, $tabla, $count);
        } catch (Exception $e) {
            echo sprintf("✗ %-30s Error\n", $tabla);
        }
    }

    echo "\n" . str_repeat("=", 50) . "\n";
    echo "BASE DE DATOS RESTAURADA EXITOSAMENTE\n";
    echo str_repeat("=", 50) . "\n";
    echo "\n";
    echo "<a href='../dashboard.php'>Ir al Dashboard</a>\n";
    echo "</pre>";

} catch (Exception $e) {
    echo "\n✗ ERROR FATAL: " . $e->getMessage() . "\n";
    echo "Stack trace:\n" . $e->getTraceAsString() . "\n";
}
