<?php
require_once '../config/db.php';
require_once '../config/functions.php';

requerirRol(['admin']);

$id = intval($_GET['id'] ?? 0);

if (!$id) {
    header('Location: index.php');
    exit;
}

// Obtener docente
$docente = query("SELECT d.*, u.usuario, u.id as usuario_id, u.estado as usuario_estado
                 FROM docentes d
                 LEFT JOIN usuarios u ON d.id_usuario = u.id
                 WHERE d.id = :id", ['id' => $id])->fetch();

if (!$docente) {
    header('Location: index.php');
    exit;
}

$mensaje = '';
$tipo_mensaje = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $dni_nuevo = trim($_POST['dni'] ?? '');
    $apellido_paterno = trim($_POST['apellido_paterno'] ?? '');
    $apellido_materno = trim($_POST['apellido_materno'] ?? '');
    $nombres = trim($_POST['nombres'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $telefono = trim($_POST['telefono'] ?? '');
    $direccion = trim($_POST['direccion'] ?? '');
    $fecha_nacimiento = $_POST['fecha_nacimiento'] ?? null;
    $genero = $_POST['genero'] ?? null;
    $estado = $_POST['estado'] ?? 'activo';

    // Validaciones
    if (empty($dni_nuevo) || empty($apellido_paterno) || empty($apellido_materno) || empty($nombres)) {
        $mensaje = 'DNI, apellidos y nombres son obligatorios';
        $tipo_mensaje = 'danger';
    } elseif (!preg_match('/^\d{8}$/', $dni_nuevo)) {
        $mensaje = 'El DNI debe tener exactamente 8 dígitos';
        $tipo_mensaje = 'danger';
    } elseif (empty($email)) {
        $mensaje = 'El email es obligatorio';
        $tipo_mensaje = 'danger';
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $mensaje = 'El email no es válido';
        $tipo_mensaje = 'danger';
    } elseif (empty($telefono)) {
        $mensaje = 'El teléfono es obligatorio';
        $tipo_mensaje = 'danger';
    } else {
        try {
            // Verificar si ya existe el DNI en otro docente
            $check_dni = query("SELECT COUNT(*) as c FROM docentes WHERE dni = :dni AND id != :id",
                              ['dni' => $dni_nuevo, 'id' => $id])->fetch();
            if ($check_dni['c'] > 0) {
                throw new Exception('Ya existe otro docente con este DNI');
            }

            // Actualizar docente
            $sql = "UPDATE docentes
                    SET dni = :dni, apellido_paterno = :ap, apellido_materno = :am, nombres = :nombres,
                        email = :email, telefono = :telefono, direccion = :direccion,
                        fecha_nacimiento = :fecha_nacimiento, genero = :genero, estado = :estado
                    WHERE id = :id";
            query($sql, [
                'dni' => $dni_nuevo,
                'ap' => $apellido_paterno,
                'am' => $apellido_materno,
                'nombres' => $nombres,
                'email' => $email,
                'telefono' => $telefono,
                'direccion' => $direccion ?: null,
                'fecha_nacimiento' => $fecha_nacimiento ?: null,
                'genero' => $genero ?: null,
                'estado' => $estado,
                'id' => $id
            ]);

            // Si cambió el DNI, actualizar también el usuario
            if ($dni_nuevo !== $docente['dni']) {
                query("UPDATE usuarios SET usuario = :usuario WHERE id = :id",
                      ['usuario' => $dni_nuevo, 'id' => $docente['usuario_id']]);
            }

            // Actualizar estado del usuario según el estado del docente
            $usuario_estado = ($estado === 'activo') ? 'activo' : 'inactivo';
            query("UPDATE usuarios SET estado = :estado WHERE id = :id",
                  ['estado' => $usuario_estado, 'id' => $docente['usuario_id']]);

            $mensaje = 'Docente actualizado exitosamente';
            $tipo_mensaje = 'success';

            // Recargar datos
            $docente = query("SELECT d.*, u.usuario, u.id as usuario_id, u.estado as usuario_estado
                             FROM docentes d
                             INNER JOIN usuarios u ON d.id_usuario = u.id
                             WHERE d.id = :id", ['id' => $id])->fetch();
        } catch (Exception $e) {
            $mensaje = 'Error al actualizar: ' . $e->getMessage();
            $tipo_mensaje = 'danger';
        }
    }
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>SICA - Editar Docente</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css" rel="stylesheet">
    <link href="../assets/css/sica.css?v=5.1" rel="stylesheet">
</head>
<body>
    <nav class="sidebar">
        <?php include '../includes/sidebar.php'; ?>
    </nav>

    <div class="main-content">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <div>
                <h3 class="mb-1">Editar Docente</h3>
                <p class="text-muted mb-0">Modificar información del docente</p>
            </div>
            <a href="index.php" class="btn btn-outline-secondary">
                <i class="bi bi-arrow-left"></i> Volver
            </a>
        </div>

        <?php if ($mensaje): ?>
            <div class="alert alert-<?= $tipo_mensaje ?> alert-dismissible fade show" role="alert">
                <?= htmlspecialchars($mensaje) ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <div class="row">
            <div class="col-lg-8">
                <div class="card">
                    <div class="card-header bg-primary text-white">
                        <h5 class="mb-0"><i class="bi bi-pencil"></i> Datos del Docente</h5>
                    </div>
                    <div class="card-body">
                        <form method="POST">
                            <div class="row">
                                <div class="col-md-4">
                                    <div class="mb-3">
                                        <label for="dni" class="form-label">DNI <span class="text-danger">*</span></label>
                                        <input type="text" class="form-control" id="dni" name="dni"
                                            value="<?= htmlspecialchars($docente['dni']) ?>"
                                            pattern="\d{8}" maxlength="8" required>
                                        <div class="form-text">También es el nombre de usuario</div>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="mb-3">
                                        <label for="apellido_paterno" class="form-label">Apellido Paterno <span class="text-danger">*</span></label>
                                        <input type="text" class="form-control" id="apellido_paterno" name="apellido_paterno"
                                            value="<?= htmlspecialchars($docente['apellido_paterno']) ?>" required>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="mb-3">
                                        <label for="apellido_materno" class="form-label">Apellido Materno <span class="text-danger">*</span></label>
                                        <input type="text" class="form-control" id="apellido_materno" name="apellido_materno"
                                            value="<?= htmlspecialchars($docente['apellido_materno']) ?>" required>
                                    </div>
                                </div>
                            </div>

                            <div class="mb-3">
                                <label for="nombres" class="form-label">Nombres <span class="text-danger">*</span></label>
                                <input type="text" class="form-control" id="nombres" name="nombres"
                                    value="<?= htmlspecialchars($docente['nombres']) ?>" required>
                            </div>

                            <div class="row">
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="email" class="form-label">Email <span class="text-danger">*</span></label>
                                        <input type="email" class="form-control" id="email" name="email" required
                                            value="<?= htmlspecialchars($docente['email'] ?? '') ?>">
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="telefono" class="form-label">Teléfono <span class="text-danger">*</span></label>
                                        <input type="text" class="form-control" id="telefono" name="telefono" required
                                            value="<?= htmlspecialchars($docente['telefono'] ?? '') ?>">
                                    </div>
                                </div>
                            </div>

                            <div class="mb-3">
                                <label for="direccion" class="form-label">Dirección</label>
                                <textarea class="form-control" id="direccion" name="direccion" rows="2"><?= htmlspecialchars($docente['direccion'] ?? '') ?></textarea>
                            </div>

                            <div class="row">
                                <div class="col-md-4">
                                    <div class="mb-3">
                                        <label for="fecha_nacimiento" class="form-label">Fecha de Nacimiento</label>
                                        <input type="date" class="form-control" id="fecha_nacimiento" name="fecha_nacimiento"
                                            value="<?= htmlspecialchars($docente['fecha_nacimiento'] ?? '') ?>">
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="mb-3">
                                        <label class="form-label">Género</label>
                                        <div>
                                            <div class="form-check form-check-inline">
                                                <input class="form-check-input" type="radio" name="genero" id="genero_m" value="M"
                                                    <?= ($docente['genero'] === 'M') ? 'checked' : '' ?>>
                                                <label class="form-check-label" for="genero_m">M</label>
                                            </div>
                                            <div class="form-check form-check-inline">
                                                <input class="form-check-input" type="radio" name="genero" id="genero_f" value="F"
                                                    <?= ($docente['genero'] === 'F') ? 'checked' : '' ?>>
                                                <label class="form-check-label" for="genero_f">F</label>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="mb-3">
                                        <label for="estado" class="form-label">Estado</label>
                                        <select class="form-select" id="estado" name="estado">
                                            <option value="activo" <?= $docente['estado'] === 'activo' ? 'selected' : '' ?>>Activo</option>
                                            <option value="inactivo" <?= $docente['estado'] === 'inactivo' ? 'selected' : '' ?>>Inactivo</option>
                                            <option value="retirado" <?= $docente['estado'] === 'retirado' ? 'selected' : '' ?>>Retirado</option>
                                        </select>
                                    </div>
                                </div>
                            </div>

                            <hr>

                            <div class="d-flex gap-2">
                                <button type="submit" class="btn btn-primary flex-grow-1">
                                    <i class="bi bi-save"></i> Actualizar Docente
                                </button>
                                <a href="index.php" class="btn btn-secondary">
                                    <i class="bi bi-x-circle"></i> Cancelar
                                </a>
                            </div>
                        </form>
                    </div>
                </div>
            </div>

            <div class="col-lg-4">
                <div class="card mb-3">
                    <div class="card-body">
                        <h6 class="card-title"><i class="bi bi-person-badge text-primary"></i> Usuario del Sistema</h6>
                        <p class="mb-1"><strong>Usuario:</strong> <?= htmlspecialchars($docente['usuario']) ?></p>
                        <p class="mb-1"><strong>Rol:</strong> Docente</p>
                        <p class="mb-0"><strong>Estado usuario:</strong>
                            <span class="badge <?= $docente['usuario_estado'] === 'activo' ? 'bg-success' : 'bg-secondary' ?>">
                                <?= ucfirst($docente['usuario_estado']) ?>
                            </span>
                        </p>
                        <hr>
                        <a href="../perfil.php" class="btn btn-sm btn-outline-primary">
                            <i class="bi bi-key"></i> Cambiar Contraseña
                        </a>
                    </div>
                </div>

                <div class="card">
                    <div class="card-body">
                        <h6 class="card-title"><i class="bi bi-info-circle text-info"></i> Información de Registro</h6>
                        <p class="small text-muted mb-1"><strong>Fecha registro:</strong>
                            <?= !empty($docente['fecha_registro']) ? date('d/m/Y H:i', strtotime($docente['fecha_registro'])) : '-' ?>
                        </p>
                        <p class="small text-muted mb-1"><strong>Último acceso:</strong>
                            <?= !empty($docente['ultimo_acceso']) ? date('d/m/Y H:i', strtotime($docente['ultimo_acceso'])) : 'Sin accesos' ?>
                        </p>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
