<?php
require_once '../config/db.php';
require_once '../config/functions.php';

requerirRol(['admin']);

$errores = [];
$exito = false;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $dni = sanitizar($_POST['dni'] ?? '');
    $nombres = sanitizar($_POST['nombres'] ?? '');
    $apellido_paterno = sanitizar($_POST['apellido_paterno'] ?? '');
    $apellido_materno = sanitizar($_POST['apellido_materno'] ?? '');
    $fecha_nacimiento = $_POST['fecha_nacimiento'] ?? '';
    $genero = $_POST['genero'] ?? '';
    $direccion = sanitizar($_POST['direccion'] ?? '');
    $telefono = sanitizar($_POST['telefono'] ?? '');

    // Validaciones
    if (empty($dni)) $errores[] = 'El DNI es obligatorio';
    elseif (!preg_match('/^\d{8}$/', $dni)) $errores[] = 'El DNI debe tener 8 dígitos';
    if (empty($nombres)) $errores[] = 'Los nombres son obligatorios';
    if (empty($apellido_paterno)) $errores[] = 'El apellido paterno es obligatorio';
    if (empty($apellido_materno)) $errores[] = 'El apellido materno es obligatorio';
    if (empty($fecha_nacimiento)) $errores[] = 'La fecha de nacimiento es obligatoria';
    if (empty($genero)) $errores[] = 'El género es obligatorio';
    if (!empty($telefono) && !preg_match('/^\d{7,15}$/', $telefono)) $errores[] = 'El teléfono debe tener entre 7 y 15 dígitos';

    // Verificar DNI duplicado en estudiantes
    $existe_estudiante = query("SELECT id FROM estudiantes WHERE dni = ?", [$dni])->fetch();
    if ($existe_estudiante) $errores[] = 'Ya existe un estudiante con este DNI';

    // Verificar DNI duplicado en usuarios
    $existe_usuario = query("SELECT id FROM usuarios WHERE usuario = ?", [$dni])->fetch();
    if ($existe_usuario) $errores[] = 'El DNI ya está registrado como usuario en el sistema';

    if (empty($errores)) {
        try {
            // Iniciar transacción
            $pdo->beginTransaction();

            // Crear usuario para el estudiante
            // Usuario = DNI, Contraseña por defecto = DNI
            $password_hash = password_hash($dni, PASSWORD_DEFAULT);
            $stmt = $pdo->prepare("INSERT INTO usuarios (usuario, password, rol, estado) VALUES (?, ?, ?, ?)");
            $stmt->execute([$dni, $password_hash, 'estudiante', 'activo']);
            $id_usuario = $pdo->lastInsertId();

            // Insertar estudiante (solo datos personales)
            $stmt = $pdo->prepare("INSERT INTO estudiantes (dni, nombres, apellido_paterno, apellido_materno, fecha_nacimiento, genero, direccion, telefono, id_usuario)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
            $stmt->execute([$dni, $nombres, $apellido_paterno, $apellido_materno, $fecha_nacimiento, $genero, $direccion, $telefono, $id_usuario]);

            // Confirmar transacción
            $pdo->commit();

            setAlerta('success', 'Estudiante registrado correctamente. Usuario creado con DNI y contraseña por defecto (DNI).');
            header('Location: index.php');
            exit;
        } catch (Exception $e) {
            // Revertir transacción en caso de error
            $pdo->rollBack();
            $errores[] = 'Error al registrar el estudiante: ' . $e->getMessage();
        }
    }
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>SICA - Nuevo Estudiante</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css" rel="stylesheet">
    <link href="../assets/css/sica.css?v=5.1" rel="stylesheet">
</head>
<body>
    <nav class="sidebar">
        <?php include '../includes/sidebar.php'; ?>
    </nav>

    <div class="main-content">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <div>
                <h3 class="mb-1">Nuevo Estudiante</h3>
                <p class="text-muted mb-0">Complete el formulario con los datos personales</p>
            </div>
            <a href="index.php" class="btn btn-outline-secondary">
                <i class="bi bi-arrow-left me-2"></i>Volver
            </a>
        </div>

        <div class="content-card">
            <?php if (!empty($errores)): ?>
                <div class="alert alert-danger">
                    <i class="bi bi-exclamation-triangle-fill me-2"></i>
                    <strong>Errores:</strong><br>
                    <?php foreach ($errores as $error): ?>
                        <?= $error ?><br>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>

            <form method="POST">
                <h5 class="mb-3 pb-2 border-bottom">Información Personal</h5>
                <div class="row g-3 mb-4">
                    <div class="col-md-3">
                        <label class="form-label">DNI <span class="text-danger">*</span></label>
                        <input type="text" name="dni" class="form-control" maxlength="8" required value="<?= htmlspecialchars($_POST['dni'] ?? '') ?>">
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">Apellido Paterno <span class="text-danger">*</span></label>
                        <input type="text" name="apellido_paterno" class="form-control" required value="<?= htmlspecialchars($_POST['apellido_paterno'] ?? '') ?>">
                    </div>
                    <div class="col-md-5">
                        <label class="form-label">Apellido Materno <span class="text-danger">*</span></label>
                        <input type="text" name="apellido_materno" class="form-control" required value="<?= htmlspecialchars($_POST['apellido_materno'] ?? '') ?>">
                    </div>
                    <div class="col-md-6">
                        <label class="form-label">Nombres <span class="text-danger">*</span></label>
                        <input type="text" name="nombres" class="form-control" required value="<?= htmlspecialchars($_POST['nombres'] ?? '') ?>">
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Fecha Nacimiento <span class="text-danger">*</span></label>
                        <input type="date" name="fecha_nacimiento" class="form-control" required value="<?= $_POST['fecha_nacimiento'] ?? '' ?>">
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Teléfono</label>
                        <input type="text" name="telefono" class="form-control" value="<?= htmlspecialchars($_POST['telefono'] ?? '') ?>" placeholder="Ej: 999999999">
                    </div>
                    <div class="col-md-6">
                        <label class="form-label">Género <span class="text-danger">*</span></label>
                        <select name="genero" class="form-select" required>
                            <option value="">Seleccione</option>
                            <option value="M" <?= (($_POST['genero'] ?? '') === 'M') ? 'selected' : '' ?>>Masculino</option>
                            <option value="F" <?= (($_POST['genero'] ?? '') === 'F') ? 'selected' : '' ?>>Femenino</option>
                        </select>
                    </div>
                    <div class="col-12">
                        <label class="form-label">Dirección</label>
                        <textarea name="direccion" class="form-control" rows="2"><?= htmlspecialchars($_POST['direccion'] ?? '') ?></textarea>
                    </div>
                </div>

                <div class="alert alert-info">
                    <i class="bi bi-info-circle me-2"></i>
                    <strong>Nota:</strong> La asignación académica (nivel, grado, sección) y la vinculación con apoderados se realizará en módulos separados.
                </div>

                <div class="d-flex gap-2">
                    <button type="submit" class="btn btn-primary">
                        <i class="bi bi-save-fill me-2"></i>Guardar Estudiante
                    </button>
                    <a href="index.php" class="btn btn-outline-secondary">Cancelar</a>
                </div>
            </form>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
