<?php
/**
 * Descargar código QR de un estudiante
 * Genera descarga directa del archivo de imagen QR
 */

require_once '../config/db.php';
require_once '../config/functions.php';

requerirRol(['admin', 'auxiliar']);

$id = intval($_GET['id'] ?? 0);
$formato = strtolower($_GET['formato'] ?? 'png');
$tamanio = intval($_GET['tamanio'] ?? 400); // Tamaño más grande para descarga

if (!$id) {
    setAlerta('danger', 'ID de estudiante no válido');
    header('Location: index.php');
    exit;
}

$estudiante = query("SELECT id, dni, nombres, apellido_paterno, apellido_materno
                    FROM estudiantes WHERE id = ?", [$id])->fetch();

if (!$estudiante) {
    setAlerta('danger', 'Estudiante no encontrado');
    header('Location: index.php');
    exit;
}

// Validar formato
if (!in_array($formato, ['png', 'jpg', 'jpeg'])) {
    $formato = 'png';
}

// Generar código QR usando API de QR Server
$qr_data = $estudiante['dni']; // El QR contiene solo el DNI
$api_url = "https://api.qrserver.com/v1/create-qr-code/?size={$tamanio}x{$tamanio}&data=" . urlencode($qr_data) . "&ecc=H&bgcolor=ffffff";

// Usar cURL para obtener la imagen
$ch = curl_init($api_url);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
curl_setopt($ch, CURLOPT_TIMEOUT, 30);
$qr_image = curl_exec($ch);
$http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
curl_close($ch);

if ($qr_image === false || $http_code != 200) {
    // Si falla la API, generar un QR simple con GD
    $qr = imagecreatetruecolor($tamanio, $tamanio);
    $blanco = imagecolorallocate($qr, 255, 255, 255);
    $negro = imagecolorallocate($qr, 0, 0, 0);
    imagefill($qr, 0, 0, $blanco);

    // Dibujar un marco
    imagerectangle($qr, 10, 10, $tamanio - 10, $tamanio - 10, $negro);

    // Texto DNI centrado
    $texto = 'DNI: ' . $estudiante['dni'];
    $font = 5;
    $x = ($tamanio - imagefontwidth($font) * strlen($texto)) / 2;
    $y = ($tamanio - imagefontheight($font)) / 2;
    imagestring($qr, $font, $x, $y, $texto, $negro);

    // Capturar la imagen GD
    ob_start();
    if ($formato === 'jpg' || $formato === 'jpeg') {
        imagejpeg($qr, null, 90);
    } else {
        imagepng($qr);
    }
    $qr_image = ob_get_clean();
    imagedestroy($qr);
}

// Nombre del archivo
$nombre_archivo = "QR_{$estudiante['dni']}_{$estudiante['apellido_paterno']}_{$estudiante['apellido_materno']}.{$formato}";
$nombre_archivo = preg_replace('/[^A-Za-z0-9_\-\.]/', '_', $nombre_archivo);

// Headers para descarga
$mime_type = ($formato === 'jpg' || $formato === 'jpeg') ? 'image/jpeg' : 'image/png';
header('Content-Type: ' . $mime_type);
header('Content-Disposition: attachment; filename="' . $nombre_archivo . '"');
header('Content-Length: ' . strlen($qr_image));
header('Cache-Control: no-cache, must-revalidate');
header('Pragma: no-cache');

echo $qr_image;
