<?php
require_once '../config/db.php';
require_once '../config/functions.php';

requerirRol(['admin']);

$id = intval($_GET['id'] ?? 0);

if (!$id) {
    setAlerta('danger', 'ID de estudiante no válido');
    header('Location: index.php');
    exit;
}

// Obtener estudiante
$estudiante = query("SELECT * FROM estudiantes WHERE id = ?", [$id])->fetch();

if (!$estudiante) {
    setAlerta('danger', 'Estudiante no encontrado');
    header('Location: index.php');
    exit;
}

$errores = [];
$exito = false;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $dni = sanitizar($_POST['dni'] ?? '');
    $nombres = sanitizar($_POST['nombres'] ?? '');
    $apellido_paterno = sanitizar($_POST['apellido_paterno'] ?? '');
    $apellido_materno = sanitizar($_POST['apellido_materno'] ?? '');
    $fecha_nacimiento = $_POST['fecha_nacimiento'] ?? '';
    $genero = $_POST['genero'] ?? '';
    $direccion = sanitizar($_POST['direccion'] ?? '');
    $telefono = sanitizar($_POST['telefono'] ?? '');
    $estado = $_POST['estado'] ?? 'activo';

    // Validaciones
    if (empty($dni)) $errores[] = 'El DNI es obligatorio';
    elseif (!preg_match('/^\d{8}$/', $dni)) $errores[] = 'El DNI debe tener 8 dígitos';
    if (empty($nombres)) $errores[] = 'Los nombres son obligatorios';
    if (empty($apellido_paterno)) $errores[] = 'El apellido paterno es obligatorio';
    if (empty($apellido_materno)) $errores[] = 'El apellido materno es obligatorio';
    if (empty($fecha_nacimiento)) $errores[] = 'La fecha de nacimiento es obligatoria';
    if (empty($genero)) $errores[] = 'El género es obligatorio';
    if (!in_array($estado, ['activo', 'retirado', 'trasladado'])) $errores[] = 'Estado no válido';
    if (!empty($telefono) && !preg_match('/^\d{7,15}$/', $telefono)) $errores[] = 'El teléfono debe tener entre 7 y 15 dígitos';

    // Verificar DNI duplicado (excluyendo el estudiante actual)
    $existe = query("SELECT id FROM estudiantes WHERE dni = ? AND id != ?", [$dni, $id])->fetch();
    if ($existe) $errores[] = 'Ya existe otro estudiante con este DNI';

    if (empty($errores)) {
        try {
            // Iniciar transacción
            $pdo->beginTransaction();

            // Si cambió el DNI, actualizar también el usuario
            if ($dni !== $estudiante['dni']) {
                // Verificar que el nuevo DNI no exista como usuario
                $existe_usuario = query("SELECT id FROM usuarios WHERE usuario = ?", [$dni])->fetch();
                if ($existe_usuario) {
                    $errores[] = 'El nuevo DNI ya está registrado como usuario';
                    throw new Exception('DNI duplicado en usuarios');
                }

                // Actualizar usuario
                query("UPDATE usuarios SET usuario = ? WHERE id = ?",
                    [$dni, $estudiante['id_usuario']]);
            }

            // Actualizar estudiante (solo datos personales)
            $sql = "UPDATE estudiantes SET
                    dni = :dni,
                    nombres = :nombres,
                    apellido_paterno = :apellido_paterno,
                    apellido_materno = :apellido_materno,
                    fecha_nacimiento = :fecha_nacimiento,
                    genero = :genero,
                    direccion = :direccion,
                    telefono = :telefono,
                    estado = :estado
                    WHERE id = :id";

            query($sql, [
                'dni' => $dni,
                'nombres' => $nombres,
                'apellido_paterno' => $apellido_paterno,
                'apellido_materno' => $apellido_materno,
                'fecha_nacimiento' => $fecha_nacimiento,
                'genero' => $genero,
                'direccion' => $direccion,
                'telefono' => $telefono,
                'estado' => $estado,
                'id' => $id
            ]);

            // Confirmar transacción
            $pdo->commit();

            setAlerta('success', 'Estudiante actualizado correctamente');
            header('Location: index.php');
            exit;
        } catch (Exception $e) {
            // Revertir transacción en caso de error
            $pdo->rollBack();
            $errores[] = 'Error al actualizar el estudiante: ' . $e->getMessage();
        }
    }
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>SICA - Editar Estudiante</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css" rel="stylesheet">
    <link href="../assets/css/sica.css?v=5.1" rel="stylesheet">
</head>
<body>
    <nav class="sidebar">
        <?php include '../includes/sidebar.php'; ?>
    </nav>

    <div class="main-content">
        <div class="row">
            <!-- Formulario de edición -->
            <div class="col-md-8">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <div>
                        <h3 class="mb-1">Editar Estudiante</h3>
                        <p class="text-muted mb-0">Modificar información personal del estudiante</p>
                    </div>
                    <a href="index.php" class="btn btn-outline-secondary">
                        <i class="bi bi-arrow-left me-2"></i>Volver
                    </a>
                </div>

                <div class="content-card">
                    <?php if (!empty($errores)): ?>
                        <div class="alert alert-danger">
                            <i class="bi bi-exclamation-triangle-fill me-2"></i>
                            <strong>Errores:</strong><br>
                            <?php foreach ($errores as $error): ?>
                                <?= $error ?><br>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>

                    <form method="POST">
                        <h5 class="mb-3 pb-2 border-bottom">Información Personal</h5>
                        <div class="row g-3 mb-4">
                            <div class="col-md-4">
                                <label class="form-label">DNI <span class="text-danger">*</span></label>
                                <input type="text" name="dni" class="form-control" maxlength="8" required value="<?= htmlspecialchars($_POST['dni'] ?? $estudiante['dni']) ?>">
                            </div>
                            <div class="col-md-8">
                                <label class="form-label">Nombres Completos</label>
                                <div="row g-2">
                                    <div class="col-4">
                                        <input type="text" name="apellido_paterno" class="form-control" placeholder="Apellido Paterno" required value="<?= htmlspecialchars($_POST['apellido_paterno'] ?? $estudiante['apellido_paterno']) ?>">
                                    </div>
                                    <div class="col-4">
                                        <input type="text" name="apellido_materno" class="form-control" placeholder="Apellido Materno" required value="<?= htmlspecialchars($_POST['apellido_materno'] ?? $estudiante['apellido_materno']) ?>">
                                    </div>
                                    <div class="col-4">
                                        <input type="text" name="nombres" class="form-control" placeholder="Nombres" required value="<?= htmlspecialchars($_POST['nombres'] ?? $estudiante['nombres']) ?>">
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <label class="form-label">Fecha Nacimiento <span class="text-danger">*</span></label>
                                <input type="date" name="fecha_nacimiento" class="form-control" required value="<?= $_POST['fecha_nacimiento'] ?? $estudiante['fecha_nacimiento'] ?>">
                            </div>
                            <div class="col-md-2">
                                <label class="form-label">Género <span class="text-danger">*</span></label>
                                <select name="genero" class="form-select" required>
                                    <option value="">Seleccione</option>
                                    <option value="M" <?= (($_POST['genero'] ?? $estudiante['genero']) === 'M' ? 'selected' : '' ) ?>>Masculino</option>
                                    <option value="F" <?= (($_POST['genero'] ?? $estudiante['genero']) === 'F' ? 'selected' : '' ) ?>>Femenino</option>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">Teléfono</label>
                                <input type="text" name="telefono" class="form-control" value="<?= htmlspecialchars($_POST['telefono'] ?? $estudiante['telefono']) ?>" placeholder="999999999">
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">Estado <span class="text-danger">*</span></label>
                                <select name="estado" class="form-select" required>
                                    <option value="activo" <?= (($_POST['estado'] ?? $estudiante['estado']) === 'activo' ? 'selected' : '' ) ?>>Activo</option>
                                    <option value="retirado" <?= (($_POST['estado'] ?? $estudiante['estado']) === 'retirado' ? 'selected' : '' ) ?>>Retirado</option>
                                    <option value="trasladado" <?= (($_POST['estado'] ?? $estudiante['estado']) === 'trasladado' ? 'selected' : '' ) ?>>Trasladado</option>
                                </select>
                            </div>
                            <div class="col-12">
                                <label class="form-label">Dirección</label>
                                <textarea name="direccion" class="form-control" rows="2"><?= htmlspecialchars($_POST['direccion'] ?? $estudiante['direccion']) ?></textarea>
                            </div>
                        </div>

                        <div class="d-flex gap-2">
                            <button type="submit" class="btn btn-warning text-white">
                                <i class="bi bi-save-fill me-2"></i>Guardar Cambios
                            </button>
                            <a href="ver.php?id=<?= $estudiante['id'] ?>" class="btn btn-info">Ver Estudiante</a>
                            <a href="index.php" class="btn btn-outline-secondary">Cancelar</a>
                        </div>
                    </form>
                </div>
            </div>

            <!-- Panel lateral con QR -->
            <div class="col-md-4">
                <div class="content-card">
                    <h5 class="mb-3 text-center">
                        <i class="bi bi-qr-code me-2"></i>Código QR
                    </h5>
                    <div class="qr-preview text-center" style="padding: 20px; background: #f8fafc; border-radius: 12px;">
                        <img src="generar_qr.php?dni=<?= urlencode($estudiante['dni']) ?>&formato=png&tamanio=200"
                             alt="Código QR del estudiante"
                             style="max-width: 200px; border: 3px solid var(--primary-color); border-radius: 10px;">
                        <p class="mt-2 mb-0 small text-muted">
                            <i class="bi bi-info-circle me-1"></i>
                            DNI: <?= htmlspecialchars($estudiante['dni']) ?>
                        </p>
                    </div>
                    <a href="../estudiantes/descargar_qr.php?id=<?= $estudiante['id'] ?>" class="btn btn-dark w-100 mt-3">
                        <i class="bi bi-download me-2"></i>Descargar QR
                    </a>
                </div>
            </div>
        </div>
    </div>

    <style>
        .qr-preview img {
            max-width: 200px;
            border: 3px solid var(--primary-color);
            border-radius: 10px;
        }
    </style>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
