<?php
require_once '../config/db.php';
require_once '../config/functions.php';
requerirAutenticacion();

$id = $_GET['id'] ?? 0;

if (!$id) {
    header('Location: index.php');
    exit;
}

// Obtener programación
$stmt = $pdo->prepare("
    SELECT
        p.id,
        p.id_seccion,
        p.id_anio_lectivo,
        p.id_tutor,
        p.id_auxiliar,
        p.capacidad,
        p.estado,
        p.observaciones,
        t.nombre as turno,
        t.abreviatura as turno_abrev,
        t.hora_ingreso,
        t.hora_salida,
        s.id_nivel,
        n.nombre AS nivel_nombre,
        s.grado,
        s.seccion,
        a.anio,
        CONCAT(d.apellido_paterno, ' ', d.apellido_materno, ', ', d.nombres) AS tutor_nombre
    FROM programaciones p
    INNER JOIN secciones s ON s.id = p.id_seccion
    INNER JOIN turnos t ON t.id = p.id_turno
    INNER JOIN niveles n ON n.id = s.id_nivel
    INNER JOIN anios_lectivos a ON a.id = p.id_anio_lectivo
    LEFT JOIN docentes d ON d.id = p.id_tutor
    WHERE p.id = ?
");
$stmt->execute([$id]);
$programacion = $stmt->fetch();

if (!$programacion) {
    header('Location: index.php');
    exit;
}

// Obtener estudiantes ya asignados
$stmt = $pdo->prepare("
    SELECT e.id, e.dni,
           CONCAT(e.apellido_paterno, ' ', e.apellido_materno, ', ', e.nombres) AS nombre,
           pe.fecha_asignacion
    FROM programaciones_estudiantes pe
    INNER JOIN estudiantes e ON e.id = pe.id_estudiante
    WHERE pe.id_programacion = ? AND pe.estado = 'activo'
    ORDER BY pe.fecha_asignacion DESC
");
$stmt->execute([$id]);
$estudiantes_asignados = $stmt->fetchAll();

// Procesar asignación
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['estudiantes'])) {
    $estudiantes_a_asignar = $_POST['estudiantes'];

    if (!empty($estudiantes_a_asignar)) {
        try {
            $pdo->beginTransaction();

            $stmt = $pdo->prepare("
                INSERT INTO programaciones_estudiantes
                (id_programacion, id_estudiante, fecha_asignacion, estado)
                VALUES (:id_programacion, :id_estudiante, CURDATE(), 'activo')
            ");

            foreach ($estudiantes_a_asignar as $id_estudiante) {
                try {
                    $stmt->execute([
                        'id_programacion' => $id,
                        'id_estudiante' => $id_estudiante
                    ]);
                } catch (PDOException $e) {
                    // Si ya existe, continuamos
                    if ($e->getCode() != 23000) {
                        throw $e;
                    }
                }
            }

            $pdo->commit();

            header("Location: ver.php?id=$id&exito_asignacion=1");
            exit;

        } catch (Exception $e) {
            $pdo->rollBack();
            $error = $e->getMessage();
        }
    }
}

// Desasignar estudiante
if (isset($_GET['desasignar'])) {
    $id_asignacion = $_GET['desasignar'];

    try {
        $stmt = $pdo->prepare("
            UPDATE programaciones_estudiantes
            SET estado = 'retirado', updated_at = NOW()
            WHERE id = ? AND id_programacion = ?
        ");
        $stmt->execute([$id_asignacion, $id]);

        header("Location: asignar_estudiantes.php?id=$id&exito_desasignacion=1");
        exit;
    } catch (Exception $e) {
        $error = $e->getMessage();
    }
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>Asignar Estudiantes - SICA</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.css" rel="stylesheet">
    <link href="../assets/css/sica.css?v=5.1" rel="stylesheet">
</head>
<body>
    <?php include '../includes/layout_elems.php'; ?>

    <div class="d-flex justify-content-between align-items-center mb-4">
            <div class="d-flex align-items-center">
                <button class="btn btn-link d-lg-none me-3" onclick="toggleSidebarMobile()">
                    <i class="bi bi-list fs-4"></i>
                </button>
                <div>
                    <nav aria-label="breadcrumb">
                        <ol class="breadcrumb mb-1">
                            <li class="breadcrumb-item"><a href="index.php" class="text-decoration-none">Programaciones</a></li>
                            <li class="breadcrumb-item"><a href="ver.php?id=<?= $id ?>" class="text-decoration-none">Ver</a></li>
                            <li class="breadcrumb-item active">Asignar Estudiantes</li>
                        </ol>
                    </nav>
                    <h3 class="mb-0">Asignar Estudiantes</h3>
                </div>
            </div>
            <a href="ver.php?id=<?= $id ?>" class="btn btn-outline-secondary">
                <i class="bi bi-arrow-left me-2"></i>Volver
            </a>
        </div>

        <?php if (isset($error)): ?>
            <div class="alert alert-danger">
                <i class="bi bi-exclamation-triangle me-2"></i><?= htmlspecialchars($error) ?>
            </div>
        <?php endif; ?>

        <?php if (isset($_GET['exito_asignacion'])): ?>
            <div class="alert alert-success">
                <i class="bi bi-check-circle me-2"></i>Estudiantes asignados correctamente.
            </div>
        <?php endif; ?>

        <?php if (isset($_GET['exito_desasignacion'])): ?>
            <div class="alert alert-success">
                <i class="bi bi-check-circle me-2"></i>Estudiante desasignado correctamente.
            </div>
        <?php endif; ?>

        <!-- Información de la Programación -->
        <div class="card border-0 shadow-sm mb-4">
            <div class="card-body">
                <div class="row">
                    <div class="col-md-3">
                        <h6 class="text-muted mb-1">Nivel</h6>
                        <h5><?= htmlspecialchars($programacion['nivel_nombre']) ?></h5>
                    </div>
                    <div class="col-md-3">
                        <h6 class="text-muted mb-1">Grado y Sección</h6>
                        <h5><?= htmlspecialchars($programacion['grado']) ?>° - <?= htmlspecialchars($programacion['seccion']) ?></h5>
                    </div>
                    <div class="col-md-3">
                        <h6 class="text-muted mb-1">Turno</h6>
                        <h5><?= ucfirst($programacion['turno']) ?></h5>
                    </div>
                    <div class="col-md-3">
                        <h6 class="text-muted mb-1">Capacidad</h6>
                        <h5><?= count($estudiantes_asignados) ?> / <?= $programacion['capacidad'] ?></h5>
                    </div>
                </div>
            </div>
        </div>

        <div class="row">
            <!-- Columna Izquierda: Estudiantes Disponibles -->
            <div class="col-lg-6">
                <div class="card border-0 shadow-sm mb-4">
                    <div class="card-header bg-primary text-white">
                        <h6 class="mb-0"><i class="bi bi-person-plus me-2"></i>Estudiantes Disponibles</h6>
                    </div>
                    <div class="card-body">
                        <div class="mb-3">
                            <input type="text" class="form-control" id="busqueda"
                                   placeholder="Buscar por DNI o nombres...">
                        </div>

                        <div id="lista-disponibles" class="list-group" style="max-height: 500px; overflow-y: auto;">
                            <div class="text-center py-4">
                                <div class="spinner-border text-primary" role="status">
                                    <span class="visually-hidden">Cargando...</span>
                                </div>
                                <p class="text-muted mt-2">Cargando estudiantes...</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Columna Derecha: Estudiantes Asignados -->
            <div class="col-lg-6">
                <div class="card border-0 shadow-sm mb-4">
                    <div class="card-header bg-success text-white">
                        <h6 class="mb-0">
                            <i class="bi bi-people me-2"></i>Estudiantes Asignados
                            <span class="badge bg-light text-dark ms-2"><?= count($estudiantes_asignados) ?></span>
                        </h6>
                    </div>
                    <div class="card-body">
                        <div class="list-group" style="max-height: 500px; overflow-y: auto;">
                            <?php if (empty($estudiantes_asignados)): ?>
                                <div class="text-center py-4">
                                    <i class="bi bi-inbox fs-1 text-muted"></i>
                                    <p class="text-muted mt-2">No hay estudiantes asignados</p>
                                </div>
                            <?php else: ?>
                                <?php foreach ($estudiantes_asignados as $est): ?>
                                    <div class="list-group-item">
                                        <div class="d-flex justify-content-between align-items-center">
                                            <div>
                                                <h6 class="mb-1"><?= htmlspecialchars($est['nombre']) ?></h6>
                                                <small class="text-muted">
                                                    <code><?= htmlspecialchars($est['dni']) ?></code>
                                                </small>
                                            </div>
                                            <div class="d-flex align-items-center">
                                                <small class="text-muted me-3">
                                                    <?= date('d/m/Y', strtotime($est['fecha_asignacion'])) ?>
                                                </small>
                                                <a href="?id=<?= $id ?>&desasignar=<?= $est['id'] ?>"
                                                   class="btn btn-sm btn-outline-danger"
                                                   onclick="return confirm('¿Desasignar estudiante?')">
                                                    <i class="bi bi-trash"></i>
                                                </a>
                                            </div>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Botón para asignar seleccionados -->
        <div class="card border-0 shadow-sm">
            <div class="card-body">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <span class="text-muted">
                            <i class="bi bi-info-circle me-2"></i>
                            Selecciona estudiantes de la lista izquierda y haz clic en "Asignar Estudiantes Seleccionados"
                        </span>
                    </div>
                    <button class="btn btn-success btn-lg" id="btnAsignar">
                        <i class="bi bi-check-circle me-2"></i>Asignar Estudiantes Seleccionados
                        <span class="badge bg-light text-dark ms-2" id="contador-seleccion">0</span>
                    </button>
                </div>
            </div>
        </div>
    </div>

    <?php include '../includes/layout_scripts.php'; ?>

    <script>
    let estudiantesDisponibles = [];
    let timeoutBusqueda = null;

    function cargarEstudiantes(busqueda = '') {
        const formData = new FormData();
        formData.append('accion', 'get_estudiantes_disponibles');
        if (busqueda) {
            formData.append('busqueda', busqueda);
        }

        fetch('ajax.php', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                estudiantesDisponibles = data.estudiantes;
                renderizarEstudiantes();
            } else {
                console.error('Error:', data.mensaje);
            }
        })
        .catch(error => console.error('Error:', error));
    }

    function renderizarEstudiantes() {
        const container = document.getElementById('lista-disponibles');

        if (estudiantesDisponibles.length === 0) {
            container.innerHTML = `
                <div class="text-center py-4">
                    <i class="bi bi-search fs-1 text-muted"></i>
                    <p class="text-muted mt-2">No se encontraron estudiantes</p>
                </div>
            `;
            return;
        }

        container.innerHTML = estudiantesDisponibles.map(est => `
            <div class="list-group-item">
                <div class="form-check">
                    <input class="form-check-input" type="checkbox"
                           name="estudiantes[]" value="${est.id}"
                           id="est_${est.id}"
                           onchange="actualizarContador()">
                    <label class="form-check-label" for="est_${est.id}">
                        <div>
                            <h6 class="mb-1">${est.nombre}</h6>
                            <small class="text-muted">
                                <code>${est.dni}</code>
                                ${est.genero === 'M' ? '♂️' : '♀️'}
                            </small>
                        </div>
                    </label>
                </div>
            </div>
        `).join('');
    }

    function actualizarContador() {
        const checkboxes = document.querySelectorAll('input[name="estudiantes[]"]:checked');
        document.getElementById('contador-seleccion').textContent = checkboxes.length;
    }

    // Event listener para búsqueda
    document.getElementById('busqueda').addEventListener('input', function(e) {
        clearTimeout(timeoutBusqueda);
        timeoutBusqueda = setTimeout(() => {
            cargarEstudiantes(e.target.value);
        }, 300);
    });

    // Asignar estudiantes
    document.getElementById('btnAsignar').addEventListener('click', function() {
        const checkboxes = document.querySelectorAll('input[name="estudiantes[]"]:checked');

        if (checkboxes.length === 0) {
            alert('Selecciona al menos un estudiante');
            return;
        }

        if (!confirm(`¿Asignar ${checkboxes.length} estudiante(s) a esta programación?`)) {
            return;
        }

        // Crear form y enviar
        const form = document.createElement('form');
        form.method = 'POST';
        form.action = '';

        checkboxes.forEach(cb => {
            const input = document.createElement('input');
            input.type = 'hidden';
            input.name = 'estudiantes[]';
            input.value = cb.value;
            form.appendChild(input);
        });

        document.body.appendChild(form);
        form.submit();
    });

    // Cargar estudiantes al inicio
    document.addEventListener('DOMContentLoaded', () => {
        cargarEstudiantes();
    });
    </script>
</body>
</html>
