<?php
require_once '../config/db.php';
require_once '../config/functions.php';

requerirAutenticacion();

// Obtener ID del estudiante de la URL
$id_estudiante = $_GET['id'] ?? null;

if (!$id_estudiante) {
    die('Error: No se proporcionó ID del estudiante');
}

$anio_lectivo_id = getAnioLectivoSeleccionado();
$anio_lectivo_nombre = $_SESSION['anio_lectivo_nombre'] ?? '';

$page_title = "Reporte de Asistencia Individual";
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?> - SICA</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.css" rel="stylesheet">
    <link href="../assets/css/sica.css?v=5.2" rel="stylesheet">
    <style>
        .alerta-critico {
            background: linear-gradient(135deg, #dc3545 0%, #c82333 100%);
            border: none;
            color: white;
        }
        .alerta-alto {
            background: linear-gradient(135deg, #fd7e14 0%, #e8590c 100%);
            border: none;
            color: white;
        }
        .alerta-medio {
            background: linear-gradient(135deg, #ffc107 0%, #e0a800 100%);
            border: none;
            color: #212529;
        }
        .alerta-bajo {
            background: linear-gradient(135deg, #17a2b8 0%, #138496 100%);
            border: none;
            color: white;
        }
        .estadistica-card {
            transition: transform 0.2s;
        }
        .estadistica-card:hover {
            transform: translateY(-5px);
        }
        .badge-estado {
            font-size: 0.85em;
            padding: 0.5em 0.7em;
        }
        .timeline-item {
            position: relative;
            padding-left: 30px;
            padding-bottom: 20px;
            border-left: 3px solid #dee2e6;
        }
        .timeline-item::before {
            content: '';
            position: absolute;
            left: -7px;
            top: 0;
            width: 12px;
            height: 12px;
            border-radius: 50%;
            background: var(--bs-primary);
        }
        .timeline-item.asistencia::before { background: #10b981; }
        .timeline-item.tardanza::before { background: #f59e0b; }
        .timeline-item.falta-justificada::before { background: #8b5cf6; }
        .timeline-item.falta-injustificada::before { background: #ef4444; }
        .progress-circle {
            width: 120px;
            height: 120px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto;
            position: relative;
        }
        .progress-circle svg {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            transform: rotate(-90deg);
        }
        .progress-circle .percentage {
            font-size: 1.5rem;
            font-weight: bold;
        }
        @media print {
            .no-print { display: none !important; }
            .print-only { display: block !important; }
            body {
                background: white !important;
                font-size: 8pt !important;
            }
            .main-content { margin: 0 !important; padding: 0 !important; }
            .card { border: none !important; box-shadow: none !important; }

            /* Encabezado - ULTRA COMPACTO */
            .reporte-header {
                border-bottom: 1px solid #1e3a8a !important;
                padding-bottom: 4px !important;
                margin-bottom: 6px !important;
                page-break-after: avoid;
                text-align: center !important;
            }
            .reporte-titulo {
                color: #1e3a8a !important;
                font-size: 9pt !important;
                font-weight: bold !important;
                margin: 0 !important;
                text-transform: uppercase;
            }
            .reporte-subtitulo {
                color: #64748b !important;
                font-size: 8pt !important;
                margin: 1px 0 0 0 !important;
                font-weight: 600 !important;
            }
            .reporte-info {
                background: #f8fafc !important;
                padding: 3px 6px !important;
                border-radius: 2px !important;
                margin-bottom: 6px !important;
                font-size: 7pt !important;
                page-break-after: avoid;
            }

            /* Secciones - COMPACTAS */
            .reporte-seccion {
                page-break-inside: avoid;
                margin-bottom: 8px !important;
            }
            .reporte-seccion-titulo {
                background: #1e3a8a !important;
                color: white !important;
                padding: 3px 6px !important;
                font-weight: bold !important;
                font-size: 8pt !important;
                margin: 0 !important;
                text-transform: uppercase;
            }
            .reporte-seccion-contenido {
                border: 1px solid #e2e8f0 !important;
                border-top: none !important;
                padding: 4px 6px !important;
                background: white !important;
            }

            /* Tablas - MUY COMPACTAS */
            .reporte-tabla {
                width: 100% !important;
                border-collapse: collapse !important;
                margin-top: 4px !important;
                font-size: 7pt !important;
            }
            .reporte-tabla th {
                background: #1e3a8a !important;
                color: white !important;
                padding: 3px 4px !important;
                text-align: left !important;
                font-weight: bold !important;
                border: 1px solid #1e3a8a !important;
                font-size: 7pt !important;
                text-transform: uppercase;
            }
            .reporte-tabla td {
                padding: 2px 4px !important;
                border: 1px solid #e2e8f0 !important;
                font-size: 7pt !important;
            }
            .reporte-tabla tr:nth-child(even) {
                background: #f8fafc !important;
            }

            /* Métricas - ULTRA COMPACTAS */
            .reporte-metricas {
                display: flex !important;
                gap: 4px !important;
                margin-top: 6px !important;
                page-break-inside: avoid;
            }
            .reporte-metrica {
                flex: 1 !important;
                background: #f8fafc !important;
                padding: 4px !important;
                text-align: center !important;
                border-left: 2px solid #1e3a8a !important;
            }
            .reporte-metrica-valor {
                font-size: 9pt !important;
                font-weight: bold !important;
                color: #1e3a8a !important;
                margin: 0 !important;
            }
            .reporte-metrica-label {
                font-size: 6pt !important;
                color: #64748b !important;
                margin: 1px 0 0 0 !important;
                text-transform: uppercase;
            }

            /* Badges - PEQUEÑOS */
            .reporte-badge {
                display: inline-block !important;
                padding: 1px 3px !important;
                border-radius: 2px !important;
                font-size: 6pt !important;
                font-weight: bold !important;
            }
            .reporte-badge-asistencia { background: #dcfce7 !important; color: #166534 !important; }
            .reporte-badge-tardanza { background: #fef3c7 !important; color: #92400e !important; }
            .reporte-badge-tardanzajustificada { background: #dbeafe !important; color: #1e40af !important; }
            .reporte-badge-falta-justificada { background: #e9d5ff !important; color: #6b21a8 !important; }
            .reporte-badge-falta-injustificada { background: #fee2e2 !important; color: #991b1b !important; }

            /* Resumen - COMPACTO */
            .reporte-resumen {
                background: #1e3a8a !important;
                color: white !important;
                padding: 6px !important;
                margin-top: 8px !important;
                page-break-inside: avoid;
            }
            .reporte-resumen-titulo {
                font-size: 8pt !important;
                font-weight: bold !important;
                margin: 0 0 4px 0 !important;
            }
            .reporte-resumen p {
                margin: 1px 0 !important;
                font-size: 7pt !important;
            }
        }
    </style>
</head>
<body>
    <?php include '../includes/layout_elems.php'; ?>

    <div class="d-flex justify-content-between align-items-center mb-4 no-print">
        <div class="d-flex align-items-center">
            <button class="btn btn-link d-lg-none me-3" onclick="toggleSidebarMobile()">
                <i class="bi bi-list fs-4"></i>
            </button>
            <div>
                <h3 class="mb-1">Reporte de Asistencia Individual</h3>
                <p class="text-muted mb-0">Análisis detallado de asistencia del estudiante</p>
            </div>
        </div>
        <div class="d-flex gap-2">
            <button class="btn btn-outline-primary" onclick="exportarPDF()">
                <i class="bi bi-file-earmark-pdf me-1"></i> PDF
            </button>
            <button class="btn btn-outline-success" onclick="exportarExcel()">
                <i class="bi bi-file-earmark-excel me-1"></i> Excel
            </button>
            <button class="btn btn-outline-secondary" onclick="window.history.back()">
                <i class="bi bi-arrow-left me-1"></i> Volver
            </button>
        </div>
    </div>

    <!-- Filtros de Periodo -->
    <div class="card mb-4 no-print">
        <div class="card-body">
            <div class="row g-3 align-items-end">
                <div class="col-md-3">
                    <label class="form-label fw-bold">Periodo Predefinido</label>
                    <select class="form-select" id="periodoPredefinido" onchange="cambiarPeriodo()">
                        <option value="hoy">Hoy</option>
                        <option value="semana">Esta semana</option>
                        <option value="mes" selected>Este mes</option>
                        <option value="trimestre">Este trimestre</option>
                        <option value="anio" selected>Año lectivo completo</option>
                        <option value="personalizado">Personalizado</option>
                    </select>
                </div>
                <div class="col-md-3">
                    <label class="form-label fw-bold">Fecha Inicio</label>
                    <input type="date" class="form-control" id="fechaInicio" onchange="cargarDatos()">
                </div>
                <div class="col-md-3">
                    <label class="form-label fw-bold">Fecha Fin</label>
                    <input type="date" class="form-control" id="fechaFin" onchange="cargarDatos()">
                </div>
                <div class="col-md-3">
                    <button class="btn btn-primary w-100" onclick="cargarDatos()">
                        <i class="bi bi-arrow-clockwise me-1"></i> Actualizar
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Loading -->
    <div id="loadingIndicator" class="text-center py-5">
        <div class="spinner-border text-primary" role="status">
            <span class="visually-hidden">Cargando...</span>
        </div>
        <p class="mt-3 text-muted">Cargando datos del estudiante...</p>
    </div>

    <!-- Versión de Impresión/PDF (oculta en pantalla) -->
    <div id="versionImpresion" class="print-only d-none">
        <!-- El contenido se generará dinámicamente con JavaScript -->
    </div>

    <!-- Contenido Principal -->
    <div id="contenidoPrincipal" style="display: none;">
        <!-- Información de la Institución -->
        <div class="card mb-3" style="background: linear-gradient(135deg, #1e3a8a 0%, #3b82f6 100%); border: none;">
            <div class="card-body text-white">
                <div class="row align-items-center">
                    <div class="col-md-2 text-center" id="institucionLogoContainer">
                        <!-- Logo se insertará aquí -->
                    </div>
                    <div class="col-md-10">
                        <h4 class="mb-1" id="institucionNombre">-</h4>
                        <p class="mb-0 opacity-75">Sistema Integrado de Convivencia y Asistencia</p>
                    </div>
                </div>
            </div>
        </div>

        <!-- Información del Estudiante -->
        <div class="card mb-4">
            <div class="card-body">
                <div class="row align-items-center">
                    <div class="col-md-8">
                        <h4 class="mb-2" id="estudianteNombre">Cargando...</h4>
                        <p class="mb-1">
                            <i class="bi bi-credit-card me-2"></i>
                            <strong>DNI:</strong> <span id="estudianteDNI">-</span>
                            <span class="mx-3">|</span>
                            <i class="bi bi-person me-2"></i>
                            <strong>Género:</strong> <span id="estudianteGenero">-</span>
                        </p>
                        <p class="mb-0">
                            <i class="bi bi-building me-2"></i>
                            <span id="estudianteNivel">-</span> -
                            <span id="estudianteGrado">-</span> -
                            <span id="estudianteSeccion">-</span>
                            <span class="mx-3">|</span>
                            <i class="bi bi-sun me-2"></i>
                            <span id="estudianteTurno">-</span>
                            <span class="mx-3">|</span>
                            <i class="bi bi-calendar3 me-2"></i>
                            <span id="estudianteAnio">-</span>
                        </p>
                    </div>
                    <div class="col-md-4 text-end">
                        <div class="progress-circle">
                            <svg>
                                <circle cx="60" cy="60" r="54" stroke="#dee2e6" stroke-width="8" fill="none"/>
                                <circle id="circleProgreso" cx="60" cy="60" r="54" stroke="#10b981" stroke-width="8" fill="none"
                                    stroke-dasharray="339.292" stroke-dashoffset="0" stroke-linecap="round"/>
                            </svg>
                            <div class="percentage">
                                <span id="porcentajeAsistencia">0</span>%
                            </div>
                        </div>
                        <p class="mt-2 mb-0 text-muted small">Asistencia Total</p>
                    </div>
                </div>
            </div>
        </div>

        <!-- Alertas -->
        <div id="alertasContainer" class="mb-4"></div>

        <!-- Estadísticas -->
        <div class="row mb-4">
            <div class="col-md-3 mb-3">
                <div class="card estadistica-card border-success">
                    <div class="card-body text-center">
                        <i class="bi bi-check-circle-fill text-success fs-1"></i>
                        <h5 class="mt-2 mb-1" id="totalAsistencias">-</h5>
                        <p class="text-muted mb-0 small">Asistencias</p>
                    </div>
                </div>
            </div>
            <div class="col-md-3 mb-3">
                <div class="card estadistica-card border-warning">
                    <div class="card-body text-center">
                        <i class="bi bi-clock-fill text-warning fs-1"></i>
                        <h5 class="mt-2 mb-1" id="totalTardanzas">-</h5>
                        <p class="text-muted mb-0 small">Tardanzas</p>
                    </div>
                </div>
            </div>
            <div class="col-md-3 mb-3">
                <div class="card estadistica-card border-info">
                    <div class="card-body text-center">
                        <i class="bi bi-calendar-check text-info fs-1"></i>
                        <h5 class="mt-2 mb-1" id="totalFaltasJustificadas">-</h5>
                        <p class="text-muted mb-0 small">Faltas Justificadas</p>
                    </div>
                </div>
            </div>
            <div class="col-md-3 mb-3">
                <div class="card estadistica-card border-danger">
                    <div class="card-body text-center">
                        <i class="bi bi-calendar-x text-danger fs-1"></i>
                        <h5 class="mt-2 mb-1" id="totalFaltasInjustificadas">-</h5>
                        <p class="text-muted mb-0 small">Faltas Injustificadas</p>
                    </div>
                </div>
            </div>
        </div>

        <!-- Métricas Adicionales -->
        <div class="row mb-4">
            <div class="col-md-6 mb-3">
                <div class="card">
                    <div class="card-body">
                        <h6 class="card-title mb-3">Métricas del Periodo</h6>
                        <div class="d-flex justify-content-between mb-2">
                            <span>Total de días registrados:</span>
                            <strong id="totalDias">-</strong>
                        </div>
                        <div class="d-flex justify-content-between mb-2">
                            <span>Días con asistencia:</span>
                            <strong id="diasAsistidos" class="text-success">-</strong>
                        </div>
                        <div class="d-flex justify-content-between mb-2">
                            <span>Minutos totales de tardanza:</span>
                            <strong id="totalMinutosTardanza">-</strong>
                        </div>
                        <div class="d-flex justify-content-between">
                            <span>Promedio tardanza por llegada tarde:</span>
                            <strong id="promedioTardanza">-</strong>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-md-6 mb-3">
                <div class="card">
                    <div class="card-body">
                        <h6 class="card-title mb-3">Porcentajes</h6>
                        <div class="mb-3">
                            <div class="d-flex justify-content-between mb-1">
                                <span>Asistencia</span>
                                <span id="lblPorcentajeAsistencia">-</span>
                            </div>
                            <div class="progress" style="height: 8px;">
                                <div id="barAsistencia" class="progress-bar bg-success" role="progressbar" style="width: 0%"></div>
                            </div>
                        </div>
                        <div class="mb-3">
                            <div class="d-flex justify-content-between mb-1">
                                <span>Faltas Injustificadas</span>
                                <span id="lblPorcentajeFaltas">-</span>
                            </div>
                            <div class="progress" style="height: 8px;">
                                <div id="barFaltas" class="progress-bar bg-danger" role="progressbar" style="width: 0%"></div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Timeline de Asistencias -->
        <div class="card">
            <div class="card-header bg-white">
                <div class="d-flex justify-content-between align-items-center">
                    <h5 class="mb-0">
                        <i class="bi bi-calendar3 me-2"></i>
                        Historial de Asistencias
                    </h5>
                    <span class="badge bg-secondary" id="totalRegistros">0 registros</span>
                </div>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-hover">
                        <thead>
                            <tr>
                                <th>Fecha</th>
                                <th>Hora Registro</th>
                                <th>Estado</th>
                                <th>Minutos Tardanza</th>
                                <th>Registrado Por</th>
                                <th>Método</th>
                                <th>Observaciones</th>
                            </tr>
                        </thead>
                        <tbody id="tablaAsistencias">
                            <tr>
                                <td colspan="7" class="text-center text-muted">
                                    No hay registros en el periodo seleccionado
                                </td>
                            </tr>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <?php include '../includes/layout_scripts.php'; ?>

    <script>
        const ID_ESTUDIANTE = <?php echo $id_estudiante; ?>;
        let reportesData = null; // Almacenar datos globalmente para impresión
        let reportesAsistencias = []; // Almacenar asistencias para impresión
        const ANIO_LECTIVO_ID = <?php echo $anio_lectivo_id; ?>;

        // Al cargar la página
        document.addEventListener('DOMContentLoaded', function() {
            inicializarFechas();
            cargarDatos();
        });

        function inicializarFechas() {
            // Establecer fechas por defecto (año lectivo completo)
            <?php
            $anio_info = query("SELECT fecha_inicio, fecha_fin FROM anios_lectivos WHERE id = ?", [$anio_lectivo_id])->fetch();
            if ($anio_info) {
            ?>
            document.getElementById('fechaInicio').value = '<?php echo $anio_info['fecha_inicio']; ?>';
            document.getElementById('fechaFin').value = '<?php echo $anio_info['fecha_fin']; ?>';
            <?php
            }
            ?>
        }

        function cambiarPeriodo() {
            const periodo = document.getElementById('periodoPredefinido').value;
            const hoy = new Date();
            let fechaInicio, fechaFin;

            switch(periodo) {
                case 'hoy':
                    fechaInicio = hoy;
                    fechaFin = hoy;
                    break;
                case 'semana':
                    const diaSemana = hoy.getDay();
                    const lunes = new Date(hoy);
                    lunes.setDate(hoy.getDate() - (diaSemana === 0 ? 6 : diaSemana - 1));
                    fechaInicio = lunes;
                    fechaFin = hoy;
                    break;
                case 'mes':
                    const primerDiaMes = new Date(hoy.getFullYear(), hoy.getMonth(), 1);
                    fechaInicio = primerDiaMes;
                    fechaFin = hoy;
                    break;
                case 'trimestre':
                    const mesActual = hoy.getMonth();
                    const inicioTrimestre = new Date(hoy.getFullYear(), Math.floor(mesActual / 3) * 3, 1);
                    fechaInicio = inicioTrimestre;
                    fechaFin = hoy;
                    break;
                case 'anio':
                    // Ya se estableció en inicializarFechas()
                    cargarDatos();
                    return;
                case 'personalizado':
                    // No hacer nada, dejar que el usuario seleccione
                    return;
            }

            document.getElementById('fechaInicio').value = formatDateForInput(fechaInicio);
            document.getElementById('fechaFin').value = formatDateForInput(fechaFin);

            cargarDatos();
        }

        function formatDateForInput(date) {
            const year = date.getFullYear();
            const month = String(date.getMonth() + 1).padStart(2, '0');
            const day = String(date.getDate()).padStart(2, '0');
            return `${year}-${month}-${day}`;
        }

        async function cargarDatos() {
            const fechaInicio = document.getElementById('fechaInicio').value;
            const fechaFin = document.getElementById('fechaFin').value;

            if (!fechaInicio || !fechaFin) {
                alert('Seleccione ambas fechas');
                return;
            }

            // Mostrar loading
            document.getElementById('loadingIndicator').style.display = 'block';
            document.getElementById('contenidoPrincipal').style.display = 'none';

            try {
                const response = await fetch(`ajax_reportes.php?accion=obtener_reporte_individual&id_estudiante=${ID_ESTUDIANTE}&fecha_inicio=${fechaInicio}&fecha_fin=${fechaFin}`);
                const data = await response.json();

                if (!data.success) {
                    throw new Error(data.mensaje);
                }

                mostrarDatos(data.data);
            } catch (error) {
                console.error('Error:', error);
                alert('Error al cargar los datos: ' + error.message);
            } finally {
                document.getElementById('loadingIndicator').style.display = 'none';
                document.getElementById('contenidoPrincipal').style.display = 'block';
            }
        }

        function mostrarDatos(data) {
            // Guardar datos globalmente para impresión
            reportesData = data;
            reportesAsistencias = data.asistencias;

            // Generar versión de impresión
            generarVersionImpresion(data);

            // Información de la institución
            const inst = data.institucion || {};
            document.getElementById('institucionNombre').textContent = inst.nombre || 'SICA';

            // Logo de la institución
            const logoContainer = document.getElementById('institucionLogoContainer');
            if (inst.logo) {
                const logoPath = inst.logo.startsWith('http') ? inst.logo : '/SICA/' + inst.logo;
                logoContainer.innerHTML = `<img src="${logoPath}" alt="Logo" style="max-height: 50px; max-width: 100%; background: white; padding: 5px; border-radius: 8px;">`;
            } else {
                logoContainer.innerHTML = '';
            }

            // Información del estudiante
            document.getElementById('estudianteNombre').textContent = data.estudiante.nombre_completo;
            document.getElementById('estudianteDNI').textContent = data.estudiante.dni;
            document.getElementById('estudianteGenero').textContent = data.estudiante.genero === 'M' ? 'Masculino' : 'Femenino';
            document.getElementById('estudianteNivel').textContent = data.estudiante.nivel;
            document.getElementById('estudianteGrado').textContent = data.estudiante.grado;
            document.getElementById('estudianteSeccion').textContent = data.estudiante.seccion;
            document.getElementById('estudianteTurno').textContent = data.estudiante.turno;
            document.getElementById('estudianteAnio').textContent = data.estudiante.anio_lectivo;

            // Estadísticas
            document.getElementById('totalAsistencias').textContent = data.estadisticas.total_asistencias;
            document.getElementById('totalTardanzas').textContent = data.estadisticas.total_tardanzas;
            document.getElementById('totalFaltasJustificadas').textContent = data.estadisticas.total_faltas_justificadas;
            document.getElementById('totalFaltasInjustificadas').textContent = data.estadisticas.total_faltas_injustificadas;

            // Métricas
            document.getElementById('totalDias').textContent = data.estadisticas.total_dias;
            document.getElementById('diasAsistidos').textContent = data.estadisticas.dias_asistidos;
            document.getElementById('totalMinutosTardanza').textContent = data.estadisticas.total_minutos_tardanza + ' min';

            const promedioTardanza = data.estadisticas.total_tardanzas > 0
                ? Math.round(data.estadisticas.total_minutos_tardanza / data.estadisticas.total_tardanzas)
                : 0;
            document.getElementById('promedioTardanza').textContent = promedioTardanza + ' min';

            // Porcentajes
            const porcentajeAsistencia = data.estadisticas.porcentaje_asistencia;
            const porcentajeFaltas = data.estadisticas.porcentaje_faltas_injustificadas;

            document.getElementById('porcentajeAsistencia').textContent = porcentajeAsistencia;
            document.getElementById('lblPorcentajeAsistencia').textContent = porcentajeAsistencia + '%';
            document.getElementById('barAsistencia').style.width = porcentajeAsistencia + '%';

            document.getElementById('lblPorcentajeFaltas').textContent = porcentajeFaltas + '%';
            document.getElementById('barFaltas').style.width = porcentajeFaltas + '%';

            // Círculo de progreso
            actualizarCirculoProgreso(porcentajeAsistencia);

            // Alertas
            mostrarAlertas(data.alertas);

            // Tabla de asistencias
            mostrarTablaAsistencias(data.asistencias);
        }

        function actualizarCirculoProgreso(porcentaje) {
            const circunferencia = 339.292;
            const offset = circunferencia - (porcentaje / 100) * circunferencia;
            const circle = document.getElementById('circleProgreso');

            circle.style.strokeDashoffset = offset;

            // Color según porcentaje
            if (porcentaje >= 90) {
                circle.setAttribute('stroke', '#10b981');
            } else if (porcentaje >= 70) {
                circle.setAttribute('stroke', '#f59e0b');
            } else {
                circle.setAttribute('stroke', '#ef4444');
            }
        }

        function mostrarAlertas(alertas) {
            const container = document.getElementById('alertasContainer');
            container.innerHTML = '';

            if (alertas.length === 0) {
                container.innerHTML = `
                    <div class="alert alert-success d-flex align-items-center" role="alert">
                        <i class="bi bi-check-circle-fill fs-4 me-3"></i>
                        <div>
                            <strong>¡Excelente!</strong> No hay alertas para este estudiante en el periodo seleccionado.
                        </div>
                    </div>
                `;
                return;
            }

            alertas.forEach(alerta => {
                const div = document.createElement('div');
                div.className = `alert alert-${alerta.tipo} d-flex align-items-start`;
                div.innerHTML = `
                    <i class="bi bi-${alerta.icono} fs-4 me-3 flex-shrink-0"></i>
                    <div class="flex-grow-1">
                        <h6 class="alert-heading mb-1">${alerta.titulo}</h6>
                        <p class="mb-1">${alerta.mensaje}</p>
                        <small class="opacity-75">
                            <i class="bi bi-lightbulb me-1"></i>
                            <strong>Acción recomendada:</strong> ${alerta.accion}
                        </small>
                    </div>
                `;
                container.appendChild(div);
            });
        }

        function mostrarTablaAsistencias(asistencias) {
            const tbody = document.getElementById('tablaAsistencias');
            document.getElementById('totalRegistros').textContent = asistencias.length + ' registros';

            if (asistencias.length === 0) {
                tbody.innerHTML = `
                    <tr>
                        <td colspan="7" class="text-center text-muted">
                            No hay registros en el periodo seleccionado
                        </td>
                    </tr>
                `;
                return;
            }

            tbody.innerHTML = asistencias.map(a => {
                const fecha = formatDate(a.fecha);
                const estadoClass = getEstadoClass(a.estado_codigo);
                const minutos = a.minutos_tardanza ? a.minutos_tardanza + ' min' : '-';
                const metodoIcon = a.metodo_registro === 'qr' ? '<i class="bi bi-qr-code-scan me-1"></i>QR' : '<i class="bi bi-keyboard me-1"></i>Manual';
                const usuario = a.registrado_por || 'Sistema';

                return `
                    <tr>
                        <td><strong>${fecha}</strong></td>
                        <td>${a.hora_registro}</td>
                        <td>
                            <span class="badge ${estadoClass}">
                                <i class="bi ${a.estado_icono} me-1"></i>
                                ${a.estado_abreviatura}
                            </span>
                        </td>
                        <td>${minutos}</td>
                        <td>${usuario}</td>
                        <td>${metodoIcon}</td>
                        <td>${a.observaciones || '-'}</td>
                    </tr>
                `;
            }).join('');
        }

        function formatDate(fechaStr) {
            const fecha = new Date(fechaStr + 'T00:00:00');
            const opciones = { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric' };
            return fecha.toLocaleDateString('es-ES', opciones);
        }

        function getEstadoClass(estado) {
            switch(estado) {
                case 'Asistencia': return 'bg-success';
                case 'Tardanza': return 'bg-warning';
                case 'Tardanza Justificada': return 'bg-info';
                case 'Falta Justificada': return 'bg-primary';
                case 'Falta Injustificada': return 'bg-danger';
                default: return 'bg-secondary';
            }
        }

        function exportarPDF() {
            if (!confirm('¿Desea generar el reporte en PDF con los datos actuales?\n\nAsegúrese de haber seleccionado el rango de fechas correcto antes de continuar.')) {
                return;
            }

            // Ocultar contenido normal y mostrar versión de impresión
            document.getElementById('contenidoPrincipal').classList.add('d-none');
            document.getElementById('versionImpresion').classList.remove('d-none');
            document.querySelectorAll('.no-print').forEach(el => el.classList.add('d-none'));

            // Imprimir
            window.print();

            // Restaurar vista normal después de imprimir
            setTimeout(() => {
                document.getElementById('contenidoPrincipal').classList.remove('d-none');
                document.getElementById('versionImpresion').classList.add('d-none');
                document.querySelectorAll('.no-print').forEach(el => el.classList.remove('d-none'));
            }, 1000);
        }

        function exportarExcel() {
            alert('Funcionalidad de exportación a Excel en desarrollo. Por ahora, use la impresión a PDF.');
        }

        function generarVersionImpresion(data) {
            const container = document.getElementById('versionImpresion');
            const inst = data.institucion || {};
            const est = data.estudiante;
            const stats = data.estadisticas;
            const periodo = data.periodo;
            const asistencias = data.asistencias;

            // Formatear fechas
            const fechaInicioFormatted = new Date(periodo.fecha_inicio + 'T00:00:00').toLocaleDateString('es-PE', { day: '2-digit', month: '2-digit', year: 'numeric' });
            const fechaFinFormatted = new Date(periodo.fecha_fin + 'T00:00:00').toLocaleDateString('es-PE', { day: '2-digit', month: '2-digit', year: 'numeric' });
            const fechaGeneracion = new Date().toLocaleDateString('es-PE', { day: '2-digit', month: '2-digit', year: 'numeric' }) + ' ' + new Date().toLocaleTimeString('es-PE', { hour: '2-digit', minute: '2-digit' });

            // Generar filas de tabla de asistencias
            let filasAsistencias = '';
            if (asistencias.length === 0) {
                filasAsistencias = '<tr><td colspan="7" class="text-center text-muted">No hay registros en el periodo seleccionado</td></tr>';
            } else {
                asistencias.forEach(a => {
                    const fechaFormatted = new Date(a.fecha + 'T00:00:00').toLocaleDateString('es-PE', { weekday: 'short', day: '2-digit', month: 'short', year: 'numeric' });
                    const horaFormatted = a.hora_registro || '--:--';

                    // Mapear estado_codigo a clase CSS
                    const badgeClassMap = {
                        'Asistencia': 'reporte-badge-asistencia',
                        'Tardanza': 'reporte-badge-tardanza',
                        'Tardanza Justificada': 'reporte-badge-tardanzajustificada',
                        'Falta Justificada': 'reporte-badge-falta-justificada',
                        'Falta Injustificada': 'reporte-badge-falta-injustificada'
                    };
                    const badgeClass = badgeClassMap[a.estado_codigo] || 'reporte-badge-asistencia';

                    const minutos = a.minutos_tardanza ? a.minutos_tardanza + ' min' : '-';
                    const metodo = a.metodo_registro === 'qr' ? 'QR' : 'Manual';
                    const usuario = a.registrado_por || 'Sistema';
                    const obs = a.observaciones || '-';

                    filasAsistencias += `
                        <tr>
                            <td>${fechaFormatted}</td>
                            <td style="text-align: center;">${horaFormatted}</td>
                            <td><span class="reporte-badge ${badgeClass}">${a.estado_nombre}</span></td>
                            <td style="text-align: center;">${minutos}</td>
                            <td>${usuario}</td>
                            <td style="text-align: center;">${metodo}</td>
                            <td>${obs}</td>
                        </tr>
                    `;
                });
            }

            // Generar alertas para impresión
            let alertasHTML = '';
            if (data.alertas && data.alertas.length > 0) {
                alertasHTML = '<div style="margin-top: 15px;">';
                data.alertas.forEach(alerta => {
                    alertasHTML += `
                        <div style="background: #fee2e2; border-left: 4px solid #dc3545; padding: 10px; margin-bottom: 10px; border-radius: 4px;">
                            <strong style="color: #991b1b;">⚠ ${alerta.titulo}</strong><br>
                            <span style="color: #7f1d1d; font-size: 9pt;">${alerta.mensaje}</span>
                        </div>
                    `;
                });
                alertasHTML += '</div>';
            }

            // Determinar color del porcentaje
            let colorPorcentaje = '#10b981'; // verde
            if (stats.porcentaje_asistencia < 70) {
                colorPorcentaje = '#ef4444'; // rojo
            } else if (stats.porcentaje_asistencia < 90) {
                colorPorcentaje = '#f59e0b'; // amarillo
            }

            // Asegurar ruta absoluta para el logo
            const logoPath = inst.logo ? (inst.logo.startsWith('http') ? inst.logo : '/SICA/' + inst.logo) : '';

            container.innerHTML = `
                <div class="reporte-header">
                    ${logoPath ? `<div style="text-align: center; margin-bottom: 15px;"><img src="${logoPath}" alt="Logo" style="max-height: 60px; max-width: 200px;"></div>` : ''}
                    <h1 class="reporte-titulo">${inst.nombre || 'SICA'}</h1>
                    <p class="reporte-subtitulo">REPORTE DE ASISTENCIA INDIVIDUAL</p>
                    <div class="reporte-info" style="margin-top: 15px;">
                        <strong>Rango de fechas:</strong> ${fechaInicioFormatted} a ${fechaFinFormatted}<br>
                        <strong>Fecha de generación:</strong> ${fechaGeneracion}
                    </div>
                </div>

                <div class="reporte-seccion" style="margin-bottom: 8px;">
                    <h3 class="reporte-seccion-titulo">DATOS DEL ESTUDIANTE</h3>
                    <div class="reporte-seccion-contenido">
                        <table class="reporte-tabla">
                            <tr>
                                <td style="width: 15%; font-weight: bold;">ALUMNO:</td>
                                <td style="width: 35%;">${est.nombre_completo}</td>
                                <td style="width: 15%; font-weight: bold;">DNI:</td>
                                <td style="width: 35%;">${est.dni}</td>
                            </tr>
                            <tr>
                                <td style="font-weight: bold;">NIVEL:</td>
                                <td>${est.nivel}</td>
                                <td style="font-weight: bold;">GRADO:</td>
                                <td>${est.grado}</td>
                            </tr>
                            <tr>
                                <td style="font-weight: bold;">SECCIÓN:</td>
                                <td>${est.seccion}</td>
                                <td style="font-weight: bold;">TURNO:</td>
                                <td>${est.turno}</td>
                            </tr>
                            <tr>
                                <td style="font-weight: bold;">AÑO LECTIVO:</td>
                                <td>${est.anio_lectivo}</td>
                                <td style="font-weight: bold;">AUXILIAR:</td>
                                <td>${est.auxiliar_nombre || 'No asignado'}</td>
                            </tr>
                            <tr>
                                <td style="font-weight: bold;">TUTOR:</td>
                                <td>${est.tutor_nombre || 'No asignado'}</td>
                                <td style="font-weight: bold;">APODERADO:</td>
                                <td>${est.apoderado_nombre || 'No asignado'}</td>
                            </tr>
                        </table>
                    </div>
                </div>

                <div class="reporte-seccion" style="margin-bottom: 8px;">
                    <h3 class="reporte-seccion-titulo">MÉTRICAS DE ASISTENCIA</h3>
                    <div class="reporte-seccion-contenido">
                        <div class="reporte-metricas">
                            <div class="reporte-metrica">
                                <p class="reporte-metrica-valor">${stats.total_dias}</p>
                                <p class="reporte-metrica-label">Total Días</p>
                            </div>
                            <div class="reporte-metrica">
                                <p class="reporte-metrica-valor">${stats.dias_asistidos}</p>
                                <p class="reporte-metrica-label">Días Asistidos</p>
                            </div>
                            <div class="reporte-metrica">
                                <p class="reporte-metrica-valor">${stats.porcentaje_asistencia}%</p>
                                <p class="reporte-metrica-label">Asistencia</p>
                            </div>
                            <div class="reporte-metrica">
                                <p class="reporte-metrica-valor">${stats.total_asistencias}</p>
                                <p class="reporte-metrica-label">Asistencias</p>
                            </div>
                            <div class="reporte-metrica">
                                <p class="reporte-metrica-valor">${stats.total_tardanzas}</p>
                                <p class="reporte-metrica-label">Tardanzas</p>
                            </div>
                            <div class="reporte-metrica">
                                <p class="reporte-metrica-valor">${stats.total_faltas_justificadas}</p>
                                <p class="reporte-metrica-label">Faltas Justificadas</p>
                            </div>
                            <div class="reporte-metrica">
                                <p class="reporte-metrica-valor">${stats.total_faltas_injustificadas}</p>
                                <p class="reporte-metrica-label">Faltas Injustificadas</p>
                            </div>
                            <div class="reporte-metrica">
                                <p class="reporte-metrica-valor">${stats.total_minutos_tardanza}</p>
                                <p class="reporte-metrica-label">Minutos Tardanza</p>
                            </div>
                        </div>
                        ${alertasHTML}
                    </div>
                </div>

                <div class="reporte-seccion" style="margin-bottom: 20px;">
                    <h3 class="reporte-seccion-titulo">DETALLE DE ASISTENCIAS</h3>
                    <div class="reporte-seccion-contenido">
                        <table class="reporte-tabla">
                            <thead>
                                <tr>
                                    <th style="width: 18%;">FECHA</th>
                                    <th style="width: 12%;">HORA</th>
                                    <th style="width: 15%;">ESTADO</th>
                                    <th style="width: 8%;">MIN.</th>
                                    <th style="width: 17%;">REGISTRADO POR</th>
                                    <th style="width: 10%;">MÉTODO</th>
                                    <th style="width: 20%;">OBSERVACIONES</th>
                                </tr>
                            </thead>
                            <tbody>
                                ${filasAsistencias}
                            </tbody>
                        </table>
                    </div>
                </div>

                <div class="reporte-resumen">
                    <div style="display: flex; gap: 15px;">
                        <div style="flex: 1;">
                            <p style="margin: 1px 0;"><strong>ASISTENCIA:</strong> ${stats.porcentaje_asistencia}%</p>
                            <p style="margin: 1px 0;"><strong>TOTAL DÍAS:</strong> ${stats.total_dias}</p>
                            <p style="margin: 1px 0;"><strong>FALTAS INJUSTIFICADAS:</strong> ${stats.total_faltas_injustificadas}</p>
                        </div>
                        <div style="flex: 1;">
                            <p style="margin: 1px 0;"><strong>TARDANZAS:</strong> ${stats.total_tardanzas}</p>
                            <p style="margin: 1px 0;"><strong>MINUTOS TARDANZA:</strong> ${stats.total_minutos_tardanza}</p>
                            <p style="margin: 1px 0;"><strong>PROMEDIO:</strong> ${stats.total_tardanzas > 0 ? Math.round(stats.total_minutos_tardanza / stats.total_tardanzas) : 0} min</p>
                        </div>
                    </div>
                </div>

                <div style="margin-top: 10px; padding-top: 8px; border-top: 1px solid #e2e8f0; text-align: center; color: #64748b; font-size: 7pt;">
                    <p style="margin: 0;">SICA - Sistema Integrado de Convivencia y Asistencia | Página 1 de 1</p>
                </div>
            `;
        }
    </script>
</body>
</html>
