<?php
// ============================================
// SCRIPT: Verificador de Ausencias
// ============================================
// Este script se ejecuta automáticamente cada 10 minutos dentro de SICA
//
// Funciones:
// 1. Detecta estudiantes sin asistencia hoy
// 2. Verifica que haya pasado el tiempo configurado desde su hora de ingreso
// 3. REGISTRA AUTOMÁTICAMENTE falta injustificada a todos
// 4. Crea notificación en el sistema
// 5. Envía correo electrónico al apoderado
//
// ============================================

// Cambiar al directorio del proyecto para que funcionen los includes
chdir(dirname(__DIR__));

require_once 'config/db.php';
require_once 'config/functions.php';

// Logging
$log_file = __DIR__ . '/logs/verificar_ausencias.log';
$log_fecha = date('Y-m-d H:i:s');

function logMessage($mensaje) {
    global $log_file, $log_fecha;
    $dir = dirname($log_file);
    if (!is_dir($dir)) {
        mkdir($dir, 0755, true);
    }
    file_put_contents($log_file, "[$log_fecha] $mensaje\n", FILE_APPEND);
    echo "[$log_fecha] $mensaje\n";
}

try {
    logMessage("=== INICIANDO VERIFICACIÓN DE AUSENCIAS ===");

    // 1. Obtener parámetros de configuración
    $config_param = query("SELECT valor FROM configuracion WHERE parametro = 'intervalo_notificacion_inasistencia'")->fetch();
    $intervalo_minutos = $config_param['valor'] ?? 90; // Por defecto 90 minutos

    logMessage("Intervalo de notificación: $intervalo_minutos minutos");

    // 2. Obtener año lectivo activo
    $anio_activo = query("SELECT id, anio FROM anios_lectivos WHERE estado = 'activo' LIMIT 1")->fetch();

    if (!$anio_activo) {
        logMessage("ERROR: No hay año lectivo activo");
        exit(1);
    }

    logMessage("Año lectivo activo: {$anio_activo['anio']}");

    // 3. Verificar que hoy es un día laboral (no feriado, no vacaciones)
    $hoy = date('Y-m-d');
    $hora_actual = date('H:i:s');

    try {
        $dia_calendario = query("SELECT * FROM calendario
                                WHERE fecha = ? AND estado = 'activo'
                                AND (aniolectivo_id IS NULL OR aniolectivo_id = ?)
                                LIMIT 1",
            [$hoy, $anio_activo['id']])->fetch();

        if ($dia_calendario && $dia_calendario['afecta_todos_niveles']) {
            logMessage("INFO: Hoy es {$dia_calendario['tipo']}: {$dia_calendario['nombre']}. No se procesan ausencias.");
            exit(0);
        }
    } catch (Exception $e) {
        // Si no existe tabla calendario, continuar
    }

    // 4. Buscar estudiantes sin asistencia hoy
    // Solo estudiantes que tengan programación activa en el año lectivo actual
    $estudiantes_sin_asistencia = query("
        SELECT DISTINCT
            e.id,
            e.dni,
            e.apellido_paterno,
            e.apellido_materno,
            e.nombres,
            pe.id_programacion,
            p.id_turno,
            t.hora_ingreso,
            t.tolerancia_minutos,
            g.grado,
            s.seccion,
            n.nombre AS nivel,
            ea.id AS id_apoderado,
            ea.email AS email_apoderado,
            ea.nombres AS nombre_apoderado,
            ea.apellido_paterno AS apoderado_paterno
        FROM estudiantes e
        INNER JOIN programaciones_estudiantes pe ON e.id = pe.id_estudiante
        INNER JOIN programaciones p ON pe.id_programacion = p.id
        INNER JOIN turnos t ON p.id_turno = t.id
        INNER JOIN secciones s ON p.id_seccion = s.id
        INNER JOIN grados g ON s.id_grado = g.id
        INNER JOIN niveles n ON g.id_nivel = n.id
        INNER JOIN estudiantes_apoderados sea ON e.id = sea.id_estudiante AND sea.id_anio_lectivo = ?
        INNER JOIN apoderados ea ON sea.id_apoderado = ea.id
        WHERE p.id_anio_lectivo = ?
        AND pe.estado = 'activo'
        AND e.estado = 'activo'
        AND p.estado = 'activo'
        AND NOT EXISTS (
            SELECT 1 FROM asistencias a
            WHERE a.id_estudiante = e.id
            AND a.fecha = ?
        )
    ", [$anio_activo['id'], $anio_activo['id'], $hoy])->fetchAll();

    logMessage("Estudiantes sin asistencia: " . count($estudiantes_sin_asistencia));

    // Obtener el estado de "Falta Injustificada"
    $estado_falta = query("SELECT id, nombre, abreviatura FROM estados_asistencia
                          WHERE nombre = 'Falta Injustificada' AND estado = 'activo'
                          LIMIT 1")->fetch();

    if (!$estado_falta) {
        logMessage("ERROR: No existe el estado 'Falta Injustificada' en estados_asistencia");
        exit(1);
    }

    $contador_faltas_registradas = 0;
    $contador_notificaciones = 0;
    $contador_correos = 0;

    // 5. Para cada estudiante, verificar si ya pasó el tiempo de notificación
    foreach ($estudiantes_sin_asistencia as $est) {
        $hora_ingreso = $est['hora_ingreso'];
        $tolerancia_minutos = $est['tolerancia_minutos'] ?? 15;

        // Calcular hora límite: hora_ingreso + intervalo_notificacion
        $hora_limite = date('H:i:s', strtotime($hora_ingreso) + ($intervalo_minutos * 60));

        // Si la hora actual es menor a la hora límite, aún no toca procesar
        if ($hora_actual < $hora_limite) {
            continue;
        }

        logMessage("PROCESANDO: {$est['apellido_paterno']} {$est['apellido_materno']}, {$est['nombres']} (DNI: {$est['dni']})");
        logMessage("  - Hora ingreso: $hora_ingreso");
        logMessage("  - Hora límite: $hora_limite");
        logMessage("  - Hora actual: $hora_actual");

        // 6. Verificar si ya tiene asistencia registrada (doble verificación)
        $ya_registrado = query("SELECT id FROM asistencias
                                WHERE id_estudiante = ? AND fecha = ?",
            [$est['id'], $hoy])->fetch();

        if ($ya_registrado) {
            logMessage("  - Ya tiene asistencia registrada. OMITIENDO.");
            continue;
        }

        // 7. REGISTRAR FALTA INJUSTIFICADA AUTOMÁTICAMENTE
        query("INSERT INTO asistencias (
                    id_estudiante, id_anio_lectivo, id_programacion, id_turno,
                    id_estado_asistencia, fecha, hora_registro, estado_codigo,
                    minutos_tardanza, registrado_por, metodo_registro)
               VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'SISTEMA', 'automatico')",
            [
                $est['id'],
                $anio_activo['id'],
                $est['id_programacion'],
                $est['id_turno'],
                $estado_falta['id'],
                $hoy,
                $hora_actual,
                $estado_falta['nombre'],
                0
            ]);

        $contador_faltas_registradas++;
        logMessage("  ✓ Falta injustificada registrada automáticamente");

        // 8. Verificar si ya existe notificación para este estudiante hoy
        $ya_notificado = query("
            SELECT id FROM notificaciones
            WHERE id_estudiante = ?
            AND tipo = 'inasistencia'
            AND DATE(fecha_notificacion) = ?
            AND id_anio_lectivo = ?
        ", [$est['id'], $hoy, $anio_activo['id']])->fetch();

        if ($ya_notificado) {
            logMessage("  - Notificación ya existe. OMITIENDO envío de correo.");
            continue;
        }

        // 9. Crear notificación en el sistema
        $titulo = "Ausencia Registrada - {$est['apellido_paterno']} {$est['apellido_materno']}, {$est['nombres']}";

        $hora_salida = date('H:i', strtotime($est['hora_ingreso']) + 3600 * 5);
        $fecha_hora_registro = date('H:i', strtotime($hora_actual));
        $fecha_notificacion = date('d/m/Y H:i');
        $fecha_hoy = date('d/m/Y');

        $mensaje = "Estimado(a) {$est['apoderado_paterno']}:\n\n";
        $mensaje .= "Se le informa que su estudiante {$est['nombres']} {$est['apellido_paterno']} {$est['apellido_materno']}\n";
        $mensaje .= "NO ha registrado asistencia hoy $fecha_hoy.\n\n";
        $mensaje .= "Se ha registrado automáticamente una FALTA INJUSTIFICADA en el sistema.\n\n";
        $mensaje .= "Datos del estudiante:\n";
        $mensaje .= "• DNI: {$est['dni']}\n";
        $mensaje .= "• Nivel: {$est['nivel']}\n";
        $mensaje .= "• Grado: {$est['grado']}\n";
        $mensaje .= "• Sección: \"{$est['seccion']}\"\n";
        $mensaje .= "• Turno: {$est['hora_ingreso']} - $hora_salida\n";
        $mensaje .= "• Hora de registro automático: $fecha_hora_registro\n\n";
        $mensaje .= "Fecha de notificación: $fecha_notificacion\n\n";
        $mensaje .= "Si esta falta es un error o puede justificarse, por favor contacte a la institución.";

        $insert_notif = query("
            INSERT INTO notificaciones (
                id_apoderado, id_estudiante, tipo, titulo, mensaje,
                id_anio_lectivo, leida, email_enviado
            ) VALUES (?, ?, 'inasistencia', ?, ?, ?, FALSE, FALSE)
        ", [
            $est['id_apoderado'],
            $est['id'],
            $titulo,
            $mensaje,
            $anio_activo['id']
        ]);

        if ($insert_notif) {
            $contador_notificaciones++;
            $notif_id = query("SELECT LAST_INSERT_ID()")->fetch()[0];
            logMessage("  ✓ Notificación creada (ID: $notif_id)");

            // 10. Enviar correo electrónico al apoderado
            if (!empty($est['email_apoderado'])) {
                $email_enviado = enviarEmailNotificacion(
                    $est['email_apoderado'],
                    $est['nombre_apoderado'] . ' ' . $est['apoderado_paterno'],
                    $titulo,
                    $mensaje
                );

                if ($email_enviado) {
                    $contador_correos++;
                    // Marcar como enviado
                    query("UPDATE notificaciones SET email_enviado = TRUE, fecha_email = NOW() WHERE id = ?", [$notif_id]);
                    logMessage("  ✓ Correo enviado a: {$est['email_apoderado']}");
                } else {
                    logMessage("  ✗ ERROR al enviar correo a: {$est['email_apoderado']}");
                }
            } else {
                logMessage("  ⚠ Apoderado sin correo electrónico configurado");
            }
        } else {
            logMessage("  ✗ ERROR al crear notificación en BD");
        }

        logMessage("");
    }

    logMessage("=== RESUMEN ===");
    logMessage("Faltas injustificadas registradas: $contador_faltas_registradas");
    logMessage("Notificaciones creadas: $contador_notificaciones");
    logMessage("Correos enviados: $contador_correos");
    logMessage("=== VERIFICACIÓN FINALIZADA ===");
    logMessage("");

} catch (Exception $e) {
    logMessage("ERROR CRÍTICO: " . $e->getMessage());
    logMessage("Stack trace: " . $e->getTraceAsString());
    exit(1);
}

/**
 * Función para enviar correo electrónico usando PHPMailer
 */
function enviarEmailNotificacion($email, $nombre_apoderado, $titulo, $mensaje) {
    try {
        // Cargar PHPMailer
        require_once __DIR__ . '/../vendor/phpmailer/phpmailer/src/Exception.php';
        require_once __DIR__ . '/../vendor/phpmailer/phpmailer/src/PHPMailer.php';
        require_once __DIR__ . '/../vendor/phpmailer/phpmailer/src/SMTP.php';

        // Obtener configuración SMTP
        $config = [];
        $cfg = query("SELECT * FROM configuracion WHERE parametro LIKE 'email_%'")->fetchAll();
        foreach ($cfg as $c) {
            $config[$c['parametro']] = $c['valor'];
        }

        if (empty($config['email_smtp_host']) || empty($config['email_smtp_user'])) {
            return false;
        }

        $mail = new PHPMailer\PHPMailer\PHPMailer(true);

        // Configuración del servidor
        $mail->isSMTP();
        $mail->Host = $config['email_smtp_host'];
        $mail->SMTPAuth = true;
        $mail->Username = $config['email_smtp_user'];
        $mail->Password = $config['email_smtp_pass'];
        $mail->SMTPSecure = PHPMailer\PHPMailer\PHPMailer::ENCRYPTION_STARTTLS;
        $mail->Port = $config['email_smtp_port'] ?? 587;

        // Remitente y destinatario
        $mail->setFrom($config['email_sistema'] ?? $config['email_smtp_user'], 'SICA - Sistema de Asistencia');
        $mail->addAddress($email, $nombre_apoderado);

        // Contenido
        $mail->isHTML(true);
        $mail->Subject = $titulo;
        $mail->Body = nl2br($mensaje);
        $mail->AltBody = strip_tags($mensaje);

        // Enviar
        $mail->send();
        return true;

    } catch (Exception $e) {
        logMessage("Error enviando email: " . $e->getMessage());
        return false;
    }
}
