<?php
/**
 * Script de Verificación del Sistema SICA
 * Ejecutar: http://localhost/SICA/verificar_sistema.php
 */
require_once 'config/db.php';

$errores = [];
$exitos = [];

// Función helper
function verificar_tabla($nombre, $campos_requeridos) {
    global $pdo, $errores, $exitos;

    try {
        $stmt = $pdo->query("DESCRIBE $nombre");
        $campos = $stmt->fetchAll(PDO::FETCH_COLUMN);

        $faltantes = [];
        foreach ($campos_requeridos as $campo) {
            if (!in_array($campo, $campos)) {
                $faltantes[] = $campo;
            }
        }

        if (empty($faltantes)) {
            $exitos[] = "✓ Tabla `$nombre` tiene todos los campos requeridos";
            return true;
        } else {
            $errores[] = "✗ Tabla `$nombre` falta campos: " . implode(', ', $faltantes);
            return false;
        }
    } catch (Exception $e) {
        $errores[] = "✗ Error al verificar tabla `$nombre`: " . $e->getMessage();
        return false;
    }
}

// Verificar tablas críticas
verificar_tabla('usuarios', ['id', 'usuario', 'password', 'rol', 'estado']);
verificar_tabla('anios_lectivos', ['id', 'anio', 'fecha_inicio', 'fecha_fin', 'estado']);
verificar_tabla('niveles', ['id', 'nombre', 'estado']);
verificar_tabla('turnos', ['id', 'nombre', 'abreviatura', 'hora_ingreso', 'hora_salida', 'tolerancia_minutos']);
verificar_tabla('grados', ['id', 'id_nivel', 'id_anio_lectivo', 'grado', 'estado']);
verificar_tabla('secciones', ['id', 'id_nivel', 'id_grado', 'id_turno', 'id_anio_lectivo', 'seccion', 'grado', 'capacidad']);
verificar_tabla('auxiliares', ['id', 'id_usuario', 'dni', 'apellido_paterno', 'apellido_materno', 'nombres', 'id_turno', 'estado']);
verificar_tabla('docentes', ['id', 'id_usuario', 'dni', 'apellido_paterno', 'apellido_materno', 'nombres']);
verificar_tabla('estudiantes', ['id', 'id_usuario', 'dni', 'apellido_paterno', 'apellido_materno', 'nombres']);
verificar_tabla('apoderados', ['id', 'id_usuario', 'dni', 'apellido_paterno', 'apellido_materno', 'nombres']);
verificar_tabla('estados_asistencia', ['id', 'nombre', 'abreviatura', 'color', 'icono', 'conteo_asistencia']);
verificar_tabla('tipos_incidencia', ['id', 'tipo', 'descripcion', 'gravedad', 'puntos_penalidad']);
verificar_tabla('areas_academicas', ['id', 'id_nivel', 'id_anio_lectivo', 'area']);
verificar_tabla('asistencias', ['id', 'id_estudiante', 'id_anio_lectivo', 'id_estado_asistencia', 'fecha', 'hora_registro']);
verificar_tabla('programaciones', ['id', 'id_seccion', 'id_anio_lectivo', 'id_tutor', 'id_auxiliar', 'id_turno']);
verificar_tabla('horarios', ['id', 'id_nivel', 'id_anio_lectivo', 'id_turno']);
verificar_tabla('calendario', ['id', 'fecha', 'tipo', 'aniolectivo_id']);
verificar_tabla('notificaciones', ['id', 'id_apoderado', 'id_estudiante', 'tipo', 'titulo', 'mensaje']);
verificar_tabla('notificaciones_email', ['id', 'id_apoderado', 'id_estudiante', 'tipo_notificacion', 'asunto', 'mensaje', 'estado_envio']);
verificar_tabla('configuracion', ['parametro', 'valor', 'descripcion']);

// Verificar datos iniciales
try {
    $anio_activo = query("SELECT COUNT(*) FROM anios_lectivos WHERE estado = 'activo'")->fetchColumn();
    if ($anio_activo > 0) {
        $exitos[] = "✓ Año lectivo activo configurado";
    } else {
        $errores[] = "✗ No hay año lectivo activo";
    }
} catch (Exception $e) {
    $errores[] = "✗ Error al verificar año lectivo: " . $e->getMessage();
}

try {
    $total_usuarios = query("SELECT COUNT(*) FROM usuarios")->fetchColumn();
    $exitos[] = "✓ Usuarios registrados: $total_usuarios";
} catch (Exception $e) {
    $errores[] = "✗ Error al contar usuarios: " . $e->getMessage();
}

try {
    $total_estados = query("SELECT COUNT(*) FROM estados_asistencia")->fetchColumn();
    $exitos[] = "✓ Estados de asistencia: $total_estados";
} catch (Exception $e) {
    $errores[] = "✗ Error al contar estados: " . $e->getMessage();
}

try {
    $total_tipos = query("SELECT COUNT(*) FROM tipos_incidencia")->fetchColumn();
    $exitos[] = "✓ Tipos de incidencia: $total_tipos";
} catch (Exception $e) {
    $errores[] = "✗ Error al contar tipos: " . $e->getMessage();
}

// Verificar usuario admin
try {
    $admin = query("SELECT * FROM usuarios WHERE usuario = 'admin'")->fetch();
    if ($admin) {
        $exitos[] = "✓ Usuario admin existe (ID: {$admin['id']})";
    } else {
        $errores[] = "✗ Usuario admin no existe";
    }
} catch (Exception $e) {
    $errores[] = "✗ Error al verificar admin: " . $e->getMessage();
}

?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>SICA - Verificación del Sistema</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css" rel="stylesheet">
</head>
<body class="bg-light">
    <div class="container py-5">
        <div class="row justify-content-center">
            <div class="col-md-8">
                <div class="card shadow">
                    <div class="card-header bg-primary text-white">
                        <h4 class="mb-0"><i class="bi bi-clipboard-check me-2"></i>Verificación del Sistema SICA</h4>
                    </div>
                    <div class="card-body">

                        <?php if (empty($errores)): ?>
                            <div class="alert alert-success">
                                <h5><i class="bi bi-check-circle-fill me-2"></i>¡Todo está correcto!</h5>
                                <p class="mb-0">El sistema está completamente instalado y configurado.</p>
                            </div>
                        <?php endif; ?>

                        <?php if (!empty($exitos)): ?>
                            <h6 class="text-success mt-3"><i class="bi bi-check-lg me-2"></i>Verificaciones Exitosas (<?= count($exitos) ?>)</h6>
                            <ul class="list-unstyled">
                                <?php foreach ($exitos as $exito): ?>
                                    <li class="text-success small mb-1"><?= $exito ?></li>
                                <?php endforeach; ?>
                            </ul>
                        <?php endif; ?>

                        <?php if (!empty($errores)): ?>
                            <h6 class="text-danger mt-3"><i class="bi bi-x-lg me-2"></i>Problemas Detectados (<?= count($errores) ?>)</h6>
                            <ul class="list-unstyled">
                                <?php foreach ($errores as $error): ?>
                                    <li class="text-danger small mb-1"><?= $error ?></li>
                                <?php endforeach; ?>
                            </ul>
                        <?php endif; ?>

                        <hr>
                        <div class="d-flex justify-content-between gap-2">
                            <a href="dashboard.php" class="btn btn-primary"><i class="bi bi-house-door me-2"></i>Ir al Dashboard</a>
                            <a href="login.php" class="btn btn-outline-secondary"><i class="bi bi-box-arrow-right me-2"></i>Cerrar Sesión</a>
                        </div>

                    </div>
                    <div class="card-footer text-muted small">
                        Fecha de verificación: <?= date('d/m/Y H:i:s') ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
